<?php

namespace Drupal\Tests\flysystem\Unit;

use Drupal\Core\StreamWrapper\StreamWrapperInterface;
use Drupal\flysystem\Flysystem\Adapter\MissingAdapter;
use Drupal\flysystem\Flysystem\Missing;
use Drupal\flysystem\FlysystemBridge;
use Drupal\flysystem\FlysystemFactory;
use Drupal\Tests\UnitTestCase;
use League\Flysystem\Filesystem;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * @coversDefaultClass \Drupal\flysystem\FlysystemBridge
 * @group flysystem
 */
class FlysystemBridgeTest extends UnitTestCase {

  /**
   * Flysystem Bridge.
   *
   * @var \Drupal\flysystem\FlysystemBridge
   */
  protected $bridge;

  /**
   * File system.
   *
   * @var \League\Flysystem\FilesystemInterface
   */
  protected $filesystem;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->bridge = new FlysystemBridge();
    $this->bridge->setStringTranslation($this->getStringTranslationStub());
    $this->bridge->setUri('test_scheme://file.txt');

    $factory = $this->prophesize(FlysystemFactory::class);
    $factory->getPlugin('test_scheme')->willReturn(new Missing());

    $this->filesystem = new Filesystem(new MissingAdapter());

    $factory->getFilesystem('test_scheme')->willReturn($this->filesystem);

    $factory->getSettings('test_scheme')->willReturn([
      'name' => '',
      'description' => '',
    ]);

    $container = new ContainerBuilder();
    $container->set('flysystem_factory', $factory->reveal());
    \Drupal::setContainer($container);
  }

  /**
   * @covers ::getType
   */
  public function testGetTypeReturnsWriteVisible() {
    $this->assertSame(StreamWrapperInterface::WRITE_VISIBLE, FlysystemBridge::getType());
  }

  /**
   * @covers ::getName
   */
  public function testGetNameFormattingCorrect() {
    $this->assertSame('Flysystem: test_scheme', (string) $this->bridge->getName());
  }

  /**
   * @covers ::getDescription
   */
  public function testGetDescriptionFormattingCorrect() {
    $this->assertSame('Flysystem: test_scheme', (string) $this->bridge->getDescription());
  }

  /**
   * @covers ::getUri
   * @covers ::setUri
   */
  public function testGetUriMatchesSetUri() {
    $this->bridge->setUri('foo://bar');
    $this->assertSame('foo://bar', $this->bridge->getUri());
  }

  /**
   * @covers ::getExternalUrl
   * @covers ::getFactory
   */
  public function testGetExternalUrlDelegatesToPlugin() {
    $this->assertSame('', $this->bridge->getExternalUrl('test_scheme://testfile.txt'));
  }

  /**
   * @covers ::realpath
   */
  public function testRealpathIsFalse() {
    $this->assertFalse($this->bridge->realpath());
  }

  /**
   * @covers ::dirname
   */
  public function testDirname() {
    $this->assertSame('test_scheme://', $this->bridge->dirname());
    $this->assertSame('test_scheme://dir://dir', $this->bridge->dirname('test_scheme:///dir://dir/file.txt'));
  }

  /**
   * @covers ::getFilesystem
   * @covers ::getFilesystemForScheme
   */
  public function testGetFilesystemOverridden() {
    $method = new \ReflectionMethod($this->bridge, 'getFilesystem');
    $method->setAccessible(TRUE);
    $this->assertSame($this->filesystem, $method->invoke($this->bridge));
  }

}
