<?php

namespace Drupal\Tests\flysystem\Unit;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Site\Settings;
use Drupal\flysystem\FlysystemBridge;
use Drupal\flysystem\FlysystemServiceProvider;
use Drupal\flysystem\PathProcessor\LocalPathProcessor;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\flysystem\FlysystemServiceProvider
 * @group flysystem
 */
class FlysystemServiceProviderTest extends UnitTestCase {

  /**
   * The container.
   *
   * @var \Symfony\Component\DependencyInjection\ContainerInterface
   */
  protected $container;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->container = new ContainerBuilder();
  }

  /**
   * @covers ::register
   */
  public function testNothingFailsIfContainerIsEmpty() {
    new Settings([]);
    (new FlysystemServiceProvider())->register($this->container);
    $this->assertFalse($this->container->has('flysystem_stream_wrapper'));
  }

  /**
   * @covers ::register
   */
  public function testMissingDriverIsSkipped() {
    new Settings(['flysystem' => ['test_scheme' => []]]);

    (new FlysystemServiceProvider())->register($this->container);

    $this->assertFalse($this->container->has('flysystem_stream_wrapper.test_scheme'));
  }

  /**
   * @covers ::register
   */
  public function testValidSchemeConfiguration() {
    new Settings(['flysystem' => ['test_scheme' => ['driver' => 'whatever']]]);

    (new FlysystemServiceProvider())->register($this->container);

    $this->assertTrue($this->container->has('flysystem_stream_wrapper.test_scheme'));
    $this->assertSame(FlysystemBridge::class, $this->container->getDefinition('flysystem_stream_wrapper.test_scheme')->getClass());
    $this->assertSame([['scheme' => 'test_scheme']], $this->container->getDefinition('flysystem_stream_wrapper.test_scheme')->getTag('stream_wrapper'));
  }

  /**
   * @covers ::register
   */
  public function testLocalRouteProviderGetsAdded() {
    new Settings([
      'flysystem' => [
        'test_scheme' => [
          'driver' => 'local',
          'config' => ['public' => TRUE],
        ],
      ],
    ]);

    (new FlysystemServiceProvider())->register($this->container);
    $this->assertSame(LocalPathProcessor::class, $this->container->getDefinition('flysystem.test_scheme.path_processor')->getClass());
  }

}
