<?php

namespace Drupal\fmm_views_edit_link\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\views\Plugin\views\field\EntityLink;
use Drupal\views\ResultRow;

/**
 * Field handler to show an edit link with a specific form mode.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("form_mode_edit_link")
 */
class FormModeEditLink extends EntityLink {

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['form_mode'] = ['default' => 'default'];
    $options['text'] = ['default' => 'Edit'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    // Get entity type - check for relationship first.
    $entity_type_id = $this->getTargetEntityType();
    $form_modes = $this->getFormModes($entity_type_id);

    $form['form_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Form mode'),
      '#description' => $this->t('Select which form mode to use for the edit link.'),
      '#options' => $form_modes,
      '#default_value' => $this->options['form_mode'],
      '#required' => TRUE,
    ];

    $form['text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Text to display'),
      '#default_value' => $this->options['text'],
      '#description' => $this->t('The text that will be displayed for the link.'),
    ];
  }

  /**
   * Get the target entity type (from relationship or base table).
   */
  protected function getTargetEntityType() {
    // First check if we're using a relationship.
    $relationship = $this->options['relationship'] ?? 'none';

    if (!empty($relationship) && $relationship !== 'none') {
      // Make sure relationships are initialized.
      if (empty($this->view->relationship)) {
        $this->view->initHandlers();
      }

      $relationship_handler = $this->view->relationship[$relationship] ?? NULL;

      if ($relationship_handler) {
        // The 'base' property points to the target entity's table.
        if (isset($relationship_handler->definition['base'])) {
          $target_table = $relationship_handler->definition['base'];
          return $this->getEntityTypeFromTable($target_table);
        }

        // Fallback: look up by base table (alternative naming).
        if (isset($relationship_handler->definition['base_table'])) {
          $target_table = $relationship_handler->definition['base_table'];
          return $this->getEntityTypeFromTable($target_table);
        }
      }
    }

    // Fallback to base entity type of the view.
    $base_entity_type = $this->view->getBaseEntityType();
    return $base_entity_type ? $base_entity_type->id() : NULL;
  }

  /**
   * Get entity type from a table name.
   */
  protected function getEntityTypeFromTable($table_name) {
    $entity_type_manager = \Drupal::entityTypeManager();
    foreach ($entity_type_manager->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type->getBaseTable() === $table_name) {
        return $entity_type_id;
      }
    }
    return NULL;
  }

  /**
   * Get available form modes for an entity type.
   */
  protected function getFormModes($entity_type_id) {
    $form_modes = ['default' => $this->t('Default')];

    if (!$entity_type_id) {
      return $form_modes;
    }

    // Get form modes from entity form mode storage.
    $form_mode_storage = \Drupal::entityTypeManager()->getStorage('entity_form_mode');
    $form_mode_entities = $form_mode_storage->loadByProperties(['targetEntityType' => $entity_type_id]);

    foreach ($form_mode_entities as $form_mode_entity) {
      $form_modes[$form_mode_entity->id()] = $form_mode_entity->label();
    }

    return $form_modes;
  }

  /**
   * Override getEntity to get the related entity when using a relationship.
   */
  public function getEntity(ResultRow $values) {
    $relationship = $this->options['relationship'] ?? 'none';

    if (!empty($relationship) && $relationship !== 'none') {
      /**
       * Get the related entity ID from the relationship data.
       * The relationship should have added the target entity ID to the row.
       */
      $entity_id_property = $relationship . '_id';

      if (isset($values->{$entity_id_property})) {
        $entity_type_id = $this->getTargetEntityType();
        $entity_id = $values->{$entity_id_property};

        return \Drupal::entityTypeManager()
          ->getStorage($entity_type_id)
          ->load($entity_id);
      }
    }

    // Fallback to parent implementation for base entity.
    return parent::getEntity($values);
  }

  /**
   * {@inheritdoc}
   */
  protected function getUrlInfo(ResultRow $row) {
    $entity = $this->getEntity($row);

    if (!$entity) {
      return NULL;
    }

    $form_mode = $this->options['form_mode'];

    if ($form_mode === 'default') {
      return $entity->toUrl('edit-form');
    }

    // For custom form modes, try Form Mode Manager routes.
    if (\Drupal::moduleHandler()->moduleExists('form_mode_manager')) {
      // Extract just mode name (in case it's entity_type.bundle.mode format).
      $form_mode_parts = explode('.', $form_mode);
      $mode_name = end($form_mode_parts);

      $route_name = "entity.{$entity->getEntityTypeId()}.edit_form.{$mode_name}";

      try {
        return Url::fromRoute($route_name, [
          $entity->getEntityTypeId() => $entity->id(),
        ]);
      } catch (\Exception $e) {
        // Fallback to default edit.
        return $entity->toUrl('edit-form');
      }
    }

    // Fallback: add form mode as query parameter.
    $url = $entity->toUrl('edit-form');
    $url->setOption('query', ['display' => $form_mode]);
    return $url;
  }

  /**
   * {@inheritdoc}
   */
  protected function renderLink(ResultRow $row) {
    $entity = $this->getEntity($row);

    if (!$entity || !$entity->access('update')) {
      return '';
    }

    $url_info = $this->getUrlInfo($row);
    if (!$url_info) {
      return '';
    }

    $text = $this->options['text'];

    // Return the rendered link as a string, not a render array.
    $link = [
      '#type' => 'link',
      '#title' => $text,
      '#url' => $url_info,
      '#attributes' => [
        'class' => ['form-mode-edit-link'],
      ],
    ];

    return \Drupal::service('renderer')->renderPlain($link);
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $this->ensureMyTable();
    $this->addAdditionalFields();
    /**
     * If we're using a relationship, the handler should already make the
     * related entity data available. We don't need to manually add fields here
     * since the relationship defines what fields are joined and available.
     */
  }

}
