<?php

namespace Drupal\formatter_suite\Plugin\Field\FieldFormatter;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\formatter_suite\Branding;

/**
 * Formats an email address.
 *
 * The email address may be shown as:
 *   - The plain text address.
 *   - The email address and a mailto link to the address.
 *   - Custom text and a mailto link to the address.
 *
 * @ingroup formatter_suite
 *
 * @FieldFormatter(
 *   id          = "formatter_suite_general_email",
 *   label       = @Translation("Formatter Suite - General Email address"),
 *   weight      = 1000,
 *   field_types = {
 *     "email",
 *   }
 * )
 */
class GeneralEmailFormatter extends FormatterBase {

  /*---------------------------------------------------------------------
   *
   * Configuration.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns an array of formatting styles.
   *
   * @return string[]
   *   Returns an associative array with internal names as keys and
   *   human-readable translated names as values.
   */
  protected static function getEmailStyles() {
    return [
      'plain'        => t('Email address as text'),
      'mailto'       => t('Email address with mail-to link'),
      'custommailto' => t('Custom text with mail-to link'),
    ];
  }

  /**
   * Returns an array of list styles.
   *
   * @return string[]
   *   Returns an associative array with internal names as keys and
   *   human-readable translated names as values.
   */
  protected static function getListStyles() {
    return [
      'span' => t('Single line list'),
      'ol'   => t('Numbered list'),
      'ul'   => t('Bulleted list'),
      'div'  => t('Non-bulleted block list'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return array_merge(
      [
        'emailStyle'    => 'mailto',
        'linkText'      => t('Email'),
        'classes'       => '',
        'listStyle'     => 'span',
        'listSeparator' => ', ',
      ],
      parent::defaultSettings());
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $this->sanitizeSettings();
    $isMultiple = $this->fieldDefinition->getFieldStorageDefinition()->isMultiple();

    $summary = [];
    switch ($this->getSetting('emailStyle')) {
      default:
      case 'plain':
        $summary[] = $this->t('Email address');
        break;

      case 'mailto':
        $summary[] = $this->t('Email address link');
        break;

      case 'custommailto':
        $summary[] = $this->t('Text with Email address link');
        break;
    }

    // If the field can store multiple values, then summarize list style.
    if ($isMultiple === TRUE) {
      $listStyles    = $this->getListStyles();
      $listStyle     = $this->getSetting('listStyle');
      $listSeparator = $this->getSetting('listSeparator');

      $text = $listStyles[$listStyle];
      if ($listStyle === 'span' && empty($listSeparator) === FALSE) {
        $text .= $this->t(', with separator');
      }
      $summary[] = $text;
    }

    return $summary;
  }

  /*---------------------------------------------------------------------
   *
   * Settings form.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns a brief description of the formatter.
   *
   * @return string
   *   Returns a brief translated description of the formatter.
   */
  protected function getDescription() {
    $isMultiple = $this->fieldDefinition->getFieldStorageDefinition()->isMultiple();
    if ($isMultiple === TRUE) {
      return $this->t("Format email addresses as plain text, or as a \"mailto\" link that starts the user's email application. Multiple field values are shown as a list on one line, bulleted, numbered, or in blocks.");
    }

    return $this->t("Format an email address as plain text, or as a \"mailto\" link that starts the user's email application.");
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $formState) {
    $this->sanitizeSettings();
    $isMultiple = $this->fieldDefinition->getFieldStorageDefinition()->isMultiple();

    // Below, some checkboxes and select choices show/hide other form
    // elements. We use Drupal's obscure 'states' feature, which adds
    // Javascript to elements to auto show/hide based upon a set of
    // simple conditions.
    //
    // Those conditions need to reference the form elements to check
    // (e.g. a checkbox), but the element ID and name are automatically
    // generated by the parent form. We cannot set them, or predict them,
    // so we cannot use them. We could use a class, but this form may be
    // shown multiple times on the same page, so a simple class would not be
    // unique. Instead, we create classes for this form only by adding a
    // random number marker to the end of the class name.
    $marker = rand();

    // Add branding.
    $elements = [];
    $elements = Branding::addFieldFormatterBranding($elements);
    $elements['#attached']['library'][] =
      'formatter_suite/formatter_suite.fieldformatter';

    // Add description.
    //
    // Use a large negative weight to insure it comes first.
    $elements['description'] = [
      '#type'          => 'html_tag',
      '#tag'           => 'div',
      '#value'         => $this->getDescription(),
      '#weight'        => -1000,
      '#attributes'    => [
        'class'        => [
          'formatter_suite-settings-description',
        ],
      ],
    ];

    $weight = 0;

    // Prompt for each setting.
    $elements['emailStyle'] = [
      '#title'         => $this->t('Link style'),
      '#type'          => 'select',
      '#options'       => $this->getEmailStyles(),
      '#default_value' => $this->getSetting('emailStyle'),
      '#weight'        => $weight++,
      '#wrapper_attributes' => [
        'class'        => [
          'formatter_suite-general-email-style',
        ],
      ],
      '#attributes'    => [
        'class'        => [
          'emailStyle-' . $marker,
        ],
      ],
    ];

    $elements['linkText'] = [
      '#title'         => $this->t('Custom text'),
      '#type'          => 'textfield',
      '#size'          => 10,
      '#default_value' => $this->getSetting('linkText'),
      '#weight'        => $weight++,
      '#wrapper_attributes' => [
        'class'        => [
          'formatter_suite-general-email-title-custom-text',
        ],
      ],
      '#states'        => [
        'visible'    => [
          '.emailStyle-' . $marker => [
            'value'    => 'custommailto',
          ],
        ],
      ],
    ];

    $elements['sectionBreak1'] = [
      '#markup' => '<div class="formatter_suite-section-break"></div>',
      '#weight' => $weight++,
    ];

    $elements['classes'] = [
      '#title'         => $this->t('Custom classes'),
      '#type'          => 'textfield',
      '#size'          => 10,
      '#default_value' => $this->getSetting('classes'),
      '#weight'        => $weight++,
      '#attributes'    => [
        'autocomplete' => 'off',
        'autocapitalize' => 'none',
        'spellcheck'   => 'false',
        'autocorrect'  => 'off',
      ],
      '#wrapper_attributes' => [
        'class'        => [
          'formatter_suite-general-email-classes',
        ],
      ],
    ];

    if ($isMultiple === TRUE) {
      $elements['sectionBreak3'] = [
        '#markup' => '<div class="formatter_suite-section-break"></div>',
        '#weight' => $weight++,
      ];

      $elements['listStyle'] = [
        '#title'         => $this->t('List style'),
        '#type'          => 'select',
        '#options'       => $this->getListStyles(),
        '#default_value' => $this->getSetting('listStyle'),
        '#weight'        => $weight++,
        '#wrapper_attributes' => [
          'class'        => [
            'formatter_suite-general-email-list-style',
          ],
        ],
        '#attributes'    => [
          'class'        => [
            'listStyle-' . $marker,
          ],
        ],
      ];

      $elements['listSeparator'] = [
        '#title'         => $this->t('Separator'),
        '#type'          => 'textfield',
        '#size'          => 10,
        '#default_value' => $this->getSetting('listSeparator'),
        '#weight'        => $weight++,
        '#attributes'    => [
          'autocomplete' => 'off',
          'autocapitalize' => 'none',
          'spellcheck'   => 'false',
          'autocorrect'  => 'off',
        ],
        '#wrapper_attributes' => [
          'class'        => [
            'formatter_suite-general-email-list-separator',
          ],
        ],
        '#states'        => [
          'visible'      => [
            '.listStyle-' . $marker => [
              'value'    => 'span',
            ],
          ],
        ],
      ];
    }

    return $elements;
  }

  /**
   * Sanitize settings to insure that they are safe and valid.
   *
   * @internal
   * Drupal's class hierarchy for plugins and their settings does not
   * include a 'validate' function, like that for other classes with forms.
   * Validation must therefore occur on use, rather than on form submission.
   * @endinternal
   */
  protected function sanitizeSettings() {
    // Get settings.
    $emailStyle = $this->getSetting('emailStyle');
    $linkText   = $this->getSetting('linkText');
    $defaults   = $this->defaultSettings();
    $isMultiple = $this->fieldDefinition->getFieldStorageDefinition()->isMultiple();

    // Sanitize & validate.
    $emailStyles = $this->getEmailStyles();
    if (empty($emailStyle) === TRUE ||
        isset($emailStyles[$emailStyle]) === FALSE) {
      $emailStyle = $defaults['emailStyle'];
      $this->setSetting('emailStyle', $emailStyle);
    }

    if (empty($linkText) === TRUE) {
      // Link text is further sanitized during use.
      $linkText = $defaults['linkText'];
      $this->setSetting('linkText', $linkText);
    }

    $listStyle = $this->getSetting('listStyle');
    $listStyles = $this->getListStyles();

    if ($isMultiple === TRUE) {
      if (empty($listStyle) === TRUE ||
          isset($listStyles[$listStyle]) === FALSE) {
        $listStyle = $defaults['listStyle'];
        $this->setSetting('listStyle', $listStyle);
      }
    }

    // Classes and custom title text are not sanitized or validated.
    // They will be added to the link, with appropriate Xss filtering.
  }

  /*---------------------------------------------------------------------
   *
   * View.
   *
   *---------------------------------------------------------------------*/

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    if ($items->isEmpty() === TRUE) {
      return [];
    }

    // Prepare custom classes, if any.
    $classes = $this->getSetting('classes');
    if (empty($classes) === TRUE) {
      $classes = [];
    }
    else {
      // Security: Class names are entered by an administrator.
      // They may not include anything but CSS-compatible words, and
      // certainly no HTML.
      //
      // Here, the class text is stripped of HTML tags as a start.
      // A regex tosses unacceptable characters in a CSS class name.
      $classes = strip_tags($classes);
      $classes = mb_ereg_replace('[^_a-zA-Z0-9- ]', '', $classes);
      if ($classes === FALSE) {
        $classes = [];
      }

      $classes = explode(' ', $classes);
    }

    $elements = [];
    foreach ($items as $delta => $item) {
      //
      // Get the link title.
      // -------------------
      // Use custom text, text from the link, or the URL. If custom text
      // is selected, but there isn't any, fall through to text from the
      // link. If there is none of that, fall through to the URL.
      switch ($this->getSetting('emailStyle')) {
        case 'custommailto':
          // Security: Custom text is entered by an administrator.
          // It may legitimately include HTML entities and minor HTML, but
          // it should not include dangerous HTML.
          //
          // Because it may include HTML, we cannot pass it directly as
          // the '#title' on a link, which will escape the HTML.
          //
          // We therefore use an Xss admin filter to remove any egreggious
          // HTML (such as scripts and styles), and then FormattableMarkup()
          // to mark the resulting text as safe.
          $title = Xss::filterAdmin($this->getSetting('linkText'));
          if (empty($title) === FALSE) {
            $title = new FormattableMarkup($title, []);
            $elements[$delta] = [
              '#type'       => 'link',
              '#title'      => $title,
              '#url'        => Url::fromUri('mailto:' . $item->value),
              '#attributes' => [
                'class'     => $classes,
              ],
            ];
            break;
          }

          // Fall-through and use the email address as the title text.
        case 'mailto':
          // Security: The email address in the field is entered by a user.
          // It needs to be a properly-formed address, and it certainly
          // should not include HTML.
          //
          // Url::fromUri() will check that the URL is well-formed and
          // throw an exception if it is not. We catch that and note the
          // problem in the rendered output.
          //
          // Including the email address in the '#title' field insures
          // that it will have any embedded HTML escaped.
          try {
            $url = Url::fromUri('mailto:' . $item->value);
            $elements[$delta] = [
              '#type'       => 'link',
              '#title'      => $item->value,
              '#url'        => $url,
              '#attributes' => [
                'class'     => $classes,
              ],
            ];
          }
          catch (\Exception $e) {
            // The URL with the email address is malformed.
            $title = $this->t(
              'Malformed email address: "@address"',
              [
                '@address' => $item->value,
              ]);
            $elements[$delta] = [
              '#type'       => 'html_tag',
              '#tag'        => 'span',
              '#value'      => $title,
              '#attributes' => [
                'class'     => $classes,
              ],
            ];
          }
          break;

        case 'plain':
          // Security: The email address in the field is entered by a user.
          // It needs to be a properly-formed address, and it certainly
          // should not include HTML.
          //
          // Including the email address in the '#title' field insures
          // that it will have any embedded HTML escaped.
          foreach ($items as $delta => $item) {
            $elements[$delta] = [
              '#type'       => 'html_tag',
              '#tag'        => 'span',
              '#value'      => $item->value,
              '#attributes' => [
                'class'     => $classes,
              ],
            ];
          }
          break;
      }
    }

    if (empty($elements) === TRUE) {
      return [];
    }

    //
    // Add multi-value field processing.
    // ---------------------------------
    // If the field has multiple values, redirect to a theme and pass
    // the list style and separator.
    $isMultiple = $this->fieldDefinition->getFieldStorageDefinition()->isMultiple();
    if ($isMultiple === TRUE) {
      // Replace the 'field' theme with one that supports lists.
      $elements['#theme'] = 'formatter_suite_field_list';

      // Set the list style.
      $elements['#list_style'] = $this->getSetting('listStyle');

      // Set the list separator.
      //
      // Security: The list separator is entered by an administrator.
      // It may legitimately include HTML entities and minor HTML, but
      // it should not include dangerous HTML.
      //
      // Because it may include HTML, we cannot pass it as-is and let a TWIG
      // template use {{ }}, which will process the text and corrupt any
      // entered HTML or HTML entities.
      //
      // We therefore use an Xss admin filter to remove any egreggious HTML
      // (such as scripts and styles), and then FormattableMarkup() to mark the
      // resulting text as safe.
      $listSeparator = Xss::filterAdmin($this->getSetting('listSeparator'));
      $elements['#list_separator'] = new FormattableMarkup($listSeparator, []);
    }

    return $elements;
  }

}
