# Form Factory Kits

#### Make custom forms... faster.

This module provides a service capable of creating `FormFactoryKit` objects.
Each kit is able to generate an array compatible with Drupal's Render API.
When appended to a `FormFactory` instance, the kit is used to help create a
complete form.

The kits provided by this module are merely covering Form API basics.
What (generic and/or client-specific) Form Factory Kits will you make?

## Examples
Concise:

```php
$factory = $this->formFactory->load($form);
$k = $this->formFactoryKitsService;

// Create a `textfield` field.
$factory->append($k->text());

// Create a `managed_file` field.
$factory->append($k->file());

// Create a `submit` button.
$factory->append($k->submit());

// Build the render array.
$form = $factory->getForm();
```

In a form:

```php
<?php

namespace Drupal\example\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\formfactory\Services\FormFactoryInterface;
use Drupal\formfactorykits\Services\FormFactoryKitsInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class ExampleForm.
 *
 * @package Drupal\example\Form
 */
class ExampleForm extends FormBase {

  private FormFactoryInterface $formFactory;

  private FormFactoryKitsInterface $formFactoryKitsService;

  public function __construct(FormFactoryInterface $formFactory,
                              FormFactoryKitsInterface $formFactoryKits)
  {
    $this->formFactory = $formFactory;
    $this->formFactoryKitsService = $formFactoryKits;
  }

  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('formfactory'),
      $container->get('formfactorykits')
    );
  }

  public function getFormId()
  {
    return 'example_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $factory = $this->formFactory->load($form);
    $k = $this->formFactoryKitsService;

    // Create VerticalTabsKit & add it to the FormFactory instance.
    $tabs = $k->verticalTabs();
    $factory->append($tabs);

    // Create a "dogs" TabKit & add it to the VerticalTabsKit instance.
    $dogsTab = $tabs->createTab('dogs')
      ->setTitle($this->t('Dogs'));
    $dogsTab->append(
      $k->image('dogs_image')
        ->setTitle($this->t('Image'))
    );
    $dogsTab->append(
      $k->textarea('dogs_description')
        ->setTitle($this->t('Description'))
    );
    $dogsTab->append(
      $k->checkboxes('dogs_attributes')
        ->setTitle($this->t('Attributes'))
        ->appendOption(['a' => $this->t('A')])
        ->appendOption(['b' => $this->t('B')])
        ->appendOption(['c' => $this->t('C')])
        ->setDefaultValue(['b'])
    );

    // Create a "cats" TabKit & add it to the VerticalTabsKit instance.
    $tabs->createTab('cats')
      ->setTitle($this->t('Cats'))
      ->append($k->image('cats_image')->setTitle($this->t('Image')))
      ->append($k->textarea('cats_description')->setTitle($this->t('Description')));

    // Create a SubmitKit & add it to the FormFactory instance.
    $factory->append($k->submit());

    // Return the modified $form render array.
    return $factory->getForm();
  }

  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    // TODO: implement method.
    throw new \BadMethodCallException();
  }

}
```

Road Map
--------
* finish implementing basic kit coverage of common Form API use cases
* integrate kits with `RenderElement` functionality where possible
* implement unit test coverage: confirm arrays generated by kits are as expected
