# Form Factory Kits

#### Make custom forms... faster.

This module provides a service capable of creating `FormFactoryKit` objects.
Each kit is able to generate an array compatible with Drupal's Render API.
When appended to a `FormFactory` instance, the kit is used to help create a
complete form.

The kits provided by this module are merely covering Form API basics.
What (generic and/or client-specific) Form Factory Kits will you make?

## Examples

### Concise
This pattern can be used to alter any `$form` array.

```php
/** @var \Drupal\formfactory\Services\FormFactoryInterface $form_factory */
$form_factory = \Drupal::service('formfactory')->load($form);
/** @var \Drupal\formfactorykits\Services\FormFactoryKitsInterface $kits */
$kits = \Drupal::service('formfactorykits');

// Create a `textfield` field.
$form_factory->append($kits->text());

// Create a `managed_file` field.
$form_factory->append($kits->file());

// Create a `submit` button.
$form_factory->append($kits->submit());

// Build the render array.
$form = $form_factory->getForm();
```

### In a form
This module provides `FormObjectBase`, which streamlines form building.
If you prefer to still extend the `FormBase` class provided by Drupal Core, you can use the pattern above in your custom form.

```php
<?php

namespace Drupal\example\Form;

use Drupal\formfactorykits\Form\FormObjectBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\formfactory\Services\FormFactoryInterface;
use Drupal\formfactorykits\Services\FormFactoryKitsInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class ExampleForm.
 *
 * @package Drupal\example\Form
 */
class ExampleForm extends FormObjectBase {

  public function getFormId()
  {
    return 'example_form';
  }

  public function buildFormObject(FormFactoryInterface $form_factory, FormFactoryKitsInterface $kits): void
  {
    // Optional: the Form State object can be accessed via the Form Factory.
    $form_state = $form_factory->getFormState();

    // Create VerticalTabsKit & add it to the FormFactory instance.
    $tabs = $kits->verticalTabs();
    $form_factory->append($tabs);

    // Create a "dogs" TabKit & add it to the VerticalTabsKit instance.
    $dogsTab = $tabs->createTab('dogs')
      ->setTitle($this->t('Dogs'));
    $dogsTab->append(
      $kits->image('dogs_image')
        ->setTitle($this->t('Image'))
    );
    $dogsTab->append(
      $kits->textarea('dogs_description')
        ->setTitle($this->t('Description'))
    );
    $dogsTab->append(
      $kits->checkboxes('dogs_attributes')
        ->setTitle($this->t('Attributes'))
        ->appendOption(['a' => $this->t('A')])
        ->appendOption(['b' => $this->t('B')])
        ->appendOption(['c' => $this->t('C')])
        ->setDefaultValue(['b'])
    );

    // Create a "cats" TabKit & add it to the VerticalTabsKit instance.
    $tabs->createTab('cats')
      ->setTitle($this->t('Cats'))
      ->append($kits->image('cats_image')->setTitle($this->t('Image')))
      ->append($kits->textarea('cats_description')->setTitle($this->t('Description')));

    // Create a SubmitKit & add it to the FormFactory instance.
    $form_factory->append($kits->submit());
  }

  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    // TODO: implement method.
    throw new \BadMethodCallException();
  }

}
```

Road Map
--------
* finish implementing basic kit coverage of common Form API use cases
* integrate kits with `RenderElement` functionality where possible
* implement unit test coverage: confirm arrays generated by kits are as expected
