<?php

namespace Drupal\fox\Plugin\FoxCommand;

use Drupal\Component\Serialization\Json;
use Drupal\fox\FoxCommandsHelper as Helper;
use Drupal\Core\Entity\EntityInterface;

/**
 * REPLACE fox command.
 *
 * @FoxCommand(
 *   id = "replace",
 *   label = @Translation("Replace fields for context record. Usage: REPLACE uField1 WITH eExpr1 [, uField2 WITH eExpr2]")
 * )
 */
class FoxCommandReplace extends FoxCommandBaseClass {

  /**
   * {@inheritdoc}
   */
  public function execute(array $params, array $variables, array $options): array {
    $entity_type = $variables['entity_type'] ?? NULL;
    if (empty($entity_type)) {
      return $this->errorReturn($this->t('Empty entity type. Use USE command.'));
    }
    if (empty($params)) {
      return $this->errorReturn($this->t('Empty params'));
    }

    $helper = $this->foxCommandsHelper();
    [$normal_params, $additional_params] = Helper::parseParameters('loop', $params);

    if (!empty($additional_params)) {
      $data = reset($additional_params);
      $data = Helper::prepareValue($data);
      $data = $helper->stringRender($data, $variables);

      if (is_array($data)) {
        $entities = $helper->getEntities($data, $variables);
        if (isset($entities['error'])) {
          return $this->errorReturn($entities['error']);
        }

        foreach ($entities as $entity) {
          $this->replace($entity, $normal_params, $variables);
        }
        $replaced = count($entities);
      }
      else {
        return $this->errorReturn($this->t('Loop data is not array'));
      }
    }
    else {
      $entity = $helper->getEntity($variables);
      if (is_array($entity) and isset($entity['error'])) {
        return $entity;
      }
      if (empty($entity)) {
        return $this->errorReturn($this->t('Empty entity'));
      }

      $result = $this->replace($entity, $normal_params, $variables);
      if (isset($result['error'])) {
        return $this->errorReturn($result['error']);
      }
      $replaced = 1;
    }

    return [
      'message' => $this->t('Replaced entities: @cnt', [
        '@cnt' => $replaced,
      ]),
    ];
  }

  /**
   * Replace entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   Entity for replacing.
   * @param array $params
   *   Parameters for replacing.
   * @param array $variables
   *   The command variables.
   */
  protected function replace(EntityInterface $entity, array $params, array $variables) {
    $helper = $this->foxCommandsHelper();
    $items = Helper::prepareParameters($params);

    $fields = $helper->getFields($entity);

    foreach ($items as $item) {
      $value = explode(' ', trim($item));
      if (count($value) < 3) {
        return $this->errorReturn($this->t('Bad REPLACE format'));
      }
      $field = array_shift($value);
      if (!array_key_exists($field, $fields)) {
        return $this->errorReturn($this->t("Wrong field '@field'", ['@field' => $field]));
      }

      $with = array_shift($value);

      $with = strtolower($with);
      if ($with !== 'with') {
        return $this->errorReturn($this->t('There is no WITH part'));
      }

      try {
        $value = Helper::prepareValue($value);
        $replace = Json::decode($value) ?? $value;
        $replace = $helper->stringRender($replace, $variables);

        $entity->set($field, $replace);
      }
      catch (\Exception $e) {
        return $this->errorReturn($e->getMessage());
      }
    }

    try {
      $entity->save();
    }
    catch (\Exception $e) {
      return $this->errorReturn($e->getMessage());
    }
  }

}
