<?php

namespace Drupal\fox\Plugin\FoxCommand;

use Drupal\fox\FoxCommandsHelper as Helper;

/**
 * FILE fox command.
 *
 * @FoxCommand(
 *   id = "file",
 *   label = @Translation("File information. Usage: FILE uri [INTO variable]")
 * )
 */
class FoxCommandFile extends FoxCommandBaseClass {

  /**
   * {@inheritdoc}
   */
  public function execute(array $params, array $variables, array $options): array {
    if (empty($params)) {
      return $this->errorReturn($this->t('Usage: FILE uri [INTO variable]'));
    }

    [$params, $into_params] = Helper::parseParameters('into', $params);
    $uri = implode(' ', $params);

    if (!empty($into_params)) {
      $variable = reset($into_params);
    }

    $file_system = \Drupal::service('file_system');
    $filepath = $file_system->realpath($uri);
    if (empty($filepath)) {
      return $this->errorReturn($this->t('Empty realpath for the uri "@uri"', [
        '@uri' => $uri,
      ]));
    }

    if (!file_exists($filepath)) {
      if (!empty($variable)) {
        return [
          'variables' => ['into' => [$variable => ['error' => TRUE]]],
        ];
      }
      else {
        return $this->errorReturn($this->t('File @filepath does not exist', [
          '@filepath' => $filepath,
        ]));
      }
    }

    $file_url_generator = \Drupal::service('file_url_generator');
    $info = [
      'realpath' => $filepath,
      'filename' => $file_system->basename($uri),
      'url' => $file_url_generator->generateAbsoluteString($uri),
      'directory' => $file_system->dirname($uri),
      'mime' => mime_content_type($filepath),
      'size' => filesize($filepath),
    ];

    if (!empty($variable)) {
      return [
        'variables' => ['into' => [$variable => $info]],
      ];
    }

    return [
      'message' => [
        'header' => array_keys($info),
        'data' => [array_values($info)],
        'horizontal' => TRUE,
      ],
    ];
  }

}
