<?php

namespace Drupal\fragments\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionableStorageInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\fragments\Entity\FragmentInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for deleting a fragment revision.
 *
 * @ingroup fragments
 */
class FragmentRevisionDeleteForm extends ConfirmFormBase {

  /**
   * The fragment revision.
   *
   * @var \Drupal\fragments\Entity\FragmentInterface
   */
  protected FragmentInterface $revision;

  /**
   * Constructs a new FragmentRevisionDeleteForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   Date formatter service.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly Connection $connection,
    protected readonly DateFormatterInterface $dateFormatter,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('database'),
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'fragment_revision_delete_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t(
      'Are you sure you want to delete the revision from %revision-date?',
      [
        '%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime()),
      ]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.fragment.version_history', ['fragment' => $this->revision->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $fragment_revision = NULL) {
    $fragmentStorage = $this->entityTypeManager->getStorage('fragment');
    assert($fragmentStorage instanceof RevisionableStorageInterface);
    $revision = $fragmentStorage->loadRevision($fragment_revision);
    assert($revision instanceof FragmentInterface);
    $this->revision = $revision;

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $fragmentStorage = $this->entityTypeManager->getStorage('fragment');
    assert($fragmentStorage instanceof RevisionableStorageInterface);
    $fragmentStorage->deleteRevision($this->revision->getRevisionId());

    $this->logger('content')->notice(
      'Fragment: deleted %title revision %revision.',
      [
        '%title' => $this->revision->label(),
        '%revision' => $this->revision->getRevisionId(),
      ]
    );

    $replacements = [
      '%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime()),
      '%title' => $this->revision->label(),
    ];
    $message = $this->t('Revision from %revision-date of fragment %title has been deleted.', $replacements);
    $this->messenger()->addMessage($message);
    $form_state->setRedirect(
      'entity.fragment.canonical',
       ['fragment' => $this->revision->id()]
    );
    if ($this->connection->query('SELECT COUNT(DISTINCT vid) FROM {fragment_field_revision} WHERE id = :id', [':id' => $this->revision->id()])->fetchField() > 1) {
      $form_state->setRedirect(
        'entity.fragment.version_history',
         ['fragment' => $this->revision->id()]
      );
    }
  }

}
