<?php

namespace Drupal\fragments\Form;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityConfirmFormBase;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Utility\Error;
use Psr\Log\LoggerAwareTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Builds the form to delete fragment type entities.
 */
class FragmentTypeDeleteForm extends EntityConfirmFormBase implements ContainerInjectionInterface {

  use LoggerAwareTrait;

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete %name?', ['%name' => $this->entity->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.fragment_type.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    try {
      $this->entity->delete();

      $this->messenger()->addMessage(
        $this->t('Fragment type %label deleted.',
          [
            '%label' => $this->entity->label(),
          ]
        )
      );
    }
    catch (EntityStorageException $e) {
      $replacements = ['%label' => $this->entity->label()];
      $this->messenger()->addError(
        $this->t('A problem occurred trying to delete fragment type %label.',
          $replacements)
      );
      Error::logException(
        $this->logger,
        $e,
        'A problem occurred trying to delete fragment type %label. %type: @message in %function (line %line of %file).',
        $replacements);
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->setLogger($container->get('logger.channel.fragments'));
    return $instance;
  }

}
