<?php

namespace Drupal\Tests\fragments\Traits;

use Drupal\fragments\Entity\Fragment;

/**
 * Provides methods to create fragment based on default settings.
 *
 * This trait is meant to be used only by test classes.
 */
trait FragmentCreationTrait {

  /**
   * Get a fragment from the database based on its title.
   *
   * @param string|\Drupal\Component\Render\MarkupInterface $title
   *   A fragment title, usually generated by $this->randomMachineName().
   * @param bool $reset
   *   (optional) Whether to reset the entity cache.
   *
   * @return \Drupal\fragments\Entity\FragmentInterface
   *   A fragment entity matching $title.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   Thrown if the fragment entity type doesn't exist.
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   *   Thrown if the storage handler couldn't be loaded.
   */
  public function getFragmentByTitle($title, $reset = FALSE) {
    $entityTypeManager = \Drupal::entityTypeManager();

    if ($reset) {
      $entityTypeManager->getStorage('fragment')->resetCache();
    }
    // Cast MarkupInterface objects to string.
    $title = (string) $title;
    $fragments = $entityTypeManager
      ->getStorage('fragment')
      ->loadByProperties(['title' => $title]);

    // Load the first fragment returned from the database.
    return reset($fragments);
  }

  /**
   * Creates a fragment based on default settings.
   *
   * @param array $settings
   *   (optional) An associative array of settings for the fragment, as used in
   *   entity_create(). Override the defaults by specifying the key and value
   *   in the array, for example:
   *   @code
   *     $this->drupalCreateFragment([
   *       'title' => t('Hello, world!'),
   *       'type' => 'article',
   *     ]);
   *   @endcode
   *   The following defaults are provided:
   *   - title: Random string.
   *   - user_id: The currently logged in user, or anonymous.
   *
   * @return \Drupal\fragments\Entity\FragmentInterface
   *   The created fragment entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   In case of failures saving the entity an exception is thrown.
   */
  protected function createFragment(array $settings = []) {
    // Populate defaults array.
    $settings += [
      'title'     => $this->randomMachineName(8),
      'user_id'       => \Drupal::currentUser()->id(),
    ];
    $fragment = Fragment::create($settings);
    $fragment->save();

    return $fragment;
  }

}
