<?php

namespace Drupal\friendship\Controller;

use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\friendship\FriendshipInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Controller\ControllerBase;
use Drupal\friendship\Ajax\OutdateMessageCommand;
use Drupal\user\UserInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Friendship controller.
 */
class FriendshipController extends ControllerBase {

  /**
   * Controller constructor.
   */
  public function __construct(
    protected FriendshipInterface $friendshipService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('friendship.friendship_service'),
    );
  }

  /**
   * General process action.
   *
   * @param string $action_name
   *   Action name.
   * @param \Drupal\user\UserInterface $target_user
   *   User id.
   * @param string $js
   *   Js indicator.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  protected function processLinkAction(string $action_name, UserInterface $target_user, string $js): Response {
    $response = new AjaxResponse();
    if ($js !== 'ajax') {
      return $response;
    }

    switch ($action_name) {
      case 'follow':
        if ($this->friendshipService->isHasRelationship($target_user)) {
          $this->friendshipService->follow($target_user);
          $response = $this->getAjaxResponse($target_user);
        }
        else {
          $response = $this->invokeOutdateMessage();
        }
        break;

      case 'unfollow':
        if ($this->friendshipService->isRequestSend($target_user)) {
          $this->friendshipService->unfollow($target_user);
          $response = $this->getAjaxResponse($target_user);
        }
        else {
          $response = $this->invokeOutdateMessage();
        }
        break;

      case 'accept':
        if ($this->friendshipService->isFollowedYou($target_user)) {
          $this->friendshipService->accept($target_user);
          $response = $this->getAjaxResponse($target_user);
        }
        else {
          $response = $this->invokeOutdateMessage();
        }
        break;

      case 'removeFriend':
        if ($this->friendshipService->isFriend($target_user)) {
          $this->friendshipService->removeFriend($target_user);
          $response = $this->getAjaxResponse($target_user);
        }
        else {
          $response = $this->invokeOutdateMessage();
        }
        break;

      case 'declineRequest':
        if ($this->friendshipService->isFollowedYou($target_user)) {
          $this->friendshipService->decline($target_user);
          return $this->redirect('page_manager.page_view_my_friends_my_friends-panels_variant-0');
        }
    }

    return $response;
  }

  /**
   * Follow user.
   *
   * @param \Drupal\user\UserInterface $user
   *   Target user.
   * @param string $js
   *   Js indicator.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  public function follow(UserInterface $user, string $js = 'nojs'): Response {
    return $this->processLinkAction('follow', $user, $js);
  }

  /**
   * Unfollow user.
   *
   * @param \Drupal\user\UserInterface $user
   *   Target user.
   * @param string $js
   *   Js indicator.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  public function unfollow(UserInterface $user, string $js = 'nojs'): Response {
    return $this->processLinkAction('unfollow', $user, $js);
  }

  /**
   * Accept user.
   *
   * @param \Drupal\user\UserInterface $user
   *   Target user.
   * @param string $js
   *   Js indicator.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  public function accept(UserInterface $user, string $js = 'nojs'): Response {
    return $this->processLinkAction('accept', $user, $js);
  }

  /**
   * Remove friend user.
   *
   * @param \Drupal\user\UserInterface $user
   *   Target user.
   * @param string $js
   *   Js indicator.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  public function removeFriend(UserInterface $user, string $js = 'nojs'): Response {
    return $this->processLinkAction('removeFriend', $user, $js);
  }

  /**
   * Decline request.
   *
   * @param \Drupal\user\UserInterface $user
   *   Target user.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  public function declineRequest(UserInterface $user): Response {
    return $this->processLinkAction('declineRequest', $user, 'ajax');
  }

  /**
   * Return ajax response for ajax link.
   *
   * @param \Drupal\user\UserInterface $target_user
   *   User object.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   Ajax response for link.
   */
  protected function getAjaxResponse(UserInterface $target_user): AjaxResponse {
    $response = new AjaxResponse();

    $link_attributes = $this->friendshipService->getProcessLink($target_user);
    $target_user_id = $target_user->id();

    $selector = '.friendship-ajax-link-' . $target_user_id;
    $response->addCommand(new ReplaceCommand($selector, $link_attributes));

    return $response;
  }

  /**
   * Invoke outdate message.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   Ajax response.
   */
  protected function invokeOutdateMessage(): AjaxResponse {
    $response = new AjaxResponse();

    $response->addCommand(new OutdateMessageCommand());
    return $response;
  }

}
