<?php

namespace Drupal\friendship;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\friendship\Entity\Friendship;
use Drupal\user\UserInterface;

/**
 * Friendship service.
 *
 * @package Drupal\friendship
 */
class FriendshipService implements FriendshipInterface {

  /**
   * Current logged user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $currentUser;

  /**
   * Configs.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Service constructor.
   */
  public function __construct(
    protected Connection $database,
    protected EntityTypeManagerInterface $entityTypeManager,
    AccountProxyInterface $currentUser,
    ConfigFactoryInterface $configFactory,
  ) {
    $userStorage = $entityTypeManager->getStorage('user');
    $this->currentUser = $userStorage->load($currentUser->id());

    $this->config = $configFactory->get('friendship.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function follow(UserInterface $target_user): void {
    $friendship = Friendship::create([
      'uid' => $this->currentUser->id(),
      'requested_uid' => $target_user->id(),
      'status' => 0,
    ]);

    $friendship->save();

    $friendship = Friendship::create([
      'uid' => $target_user->id(),
      'requested_uid' => $this->currentUser->id(),
      'status' => -1,
    ]);

    $friendship->save();
  }

  /**
   * {@inheritdoc}
   */
  public function unfollow(UserInterface $target_user): void {
    $entity_ids = $this->getFriendshipEntityIds($target_user);
    if (empty($entity_ids)) {
      return;
    }

    $storage_handler = $this->entityTypeManager->getStorage('friendship');
    $entities = $storage_handler->loadMultiple($entity_ids);

    if ($entities) {
      $storage_handler->delete($entities);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function accept(UserInterface $target_user): void {
    $entity_ids = $this->getFriendshipEntityIds($target_user);
    if (empty($entity_ids)) {
      return;
    }

    $storage_handler = $this->entityTypeManager->getStorage('friendship');
    $entities = $storage_handler->loadMultiple($entity_ids);

    /** @var \Drupal\friendship\Entity\Friendship $friendship */
    foreach ($entities as $friendship) {
      $friendship->set('status', 1);
      $friendship->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function removeFriend(UserInterface $target_user): void {
    $storage_handler = $this->entityTypeManager->getStorage('friendship');

    // Delete friends and made subscribers.
    $query = $storage_handler->getQuery();
    $query->condition('uid', $this->currentUser->id())
      ->condition('requested_uid', $target_user->id())
      ->accessCheck(FALSE);

    $result = $query->execute();

    $entity_id = array_shift($result);
    $friendship = $storage_handler->load($entity_id);

    if ($friendship) {
      $friendship->set('status', -1);
      $friendship->save();
    }

    $query = $storage_handler->getQuery();
    $query->condition('uid', $target_user->id())
      ->condition('requested_uid', $this->currentUser->id())
      ->accessCheck(FALSE);

    $result = $query->execute();

    $entity_id = array_shift($result);
    $friendship = $storage_handler->load($entity_id);

    if ($friendship) {
      $friendship->set('status', 0);
      $friendship->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function decline(UserInterface $target_user): void {
    $entity_ids = $this->getFriendshipEntityIds($target_user);
    if (empty($entity_ids)) {
      return;
    }

    $storage_handler = $this->entityTypeManager->getStorage('friendship');
    $entities = $storage_handler->loadMultiple($entity_ids);

    if ($entities) {
      $storage_handler->delete($entities);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isRequestSend(UserInterface $target_user): bool {
    $result = $this->getFriendshipRow($this->currentUser, $target_user);

    // @todo made it more elegance.
    if (isset($result[0]->status) && $result[0]->status == 0) {
      $result = $this->getFriendshipRow($target_user, $this->currentUser);

      if (isset($result[0]->status) && $result[0]->status == -1) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isFriend(UserInterface $target_user): bool {
    $result = $this->getFriendshipRow($this->currentUser, $target_user);

    // @todo made it more elegance.
    if (isset($result[0]->status) && $result[0]->status == 1) {
      $result = $this->getFriendshipRow($target_user, $this->currentUser);
      // @todo made it more elegance.
      if (isset($result[0]->status) && $result[0]->status == 1) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isFollowedYou(UserInterface $target_user): bool {
    $result = $this->getFriendshipRow($this->currentUser, $target_user);

    // @todo made it more elegance.
    if (isset($result[0]->status) && $result[0]->status == -1) {
      $result = $this->getFriendshipRow($target_user, $this->currentUser);
      // @todo made it more elegance.
      if (isset($result[0]->status) && $result[0]->status == 0) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isHasRelationship(UserInterface $target_user): bool {
    $result = $this->getFriendshipRow($this->currentUser, $target_user);

    if (empty($result)) {
      $result = $this->getFriendshipRow($target_user, $this->currentUser);

      if (empty($result)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Select all friends rows.
   */
  protected function getFriendshipRow(UserInterface $current_user, UserInterface $target_user): array {
    $query = $this->database->select('friendship', 'fr')
      ->fields('fr', ['status'])
      ->condition('fr.uid', $current_user->id())
      ->condition('fr.requested_uid', $target_user->id());

    return $query->execute()->fetchAll();
  }

  /**
   * Get friendship entity_id by the requested user.
   *
   * @param \Drupal\user\Entity\User $target_user
   *   Target user.
   *
   * @return array
   *   Entity ids.
   */
  protected function getFriendshipEntityIds(UserInterface $target_user): array {
    $query = $this->entityTypeManager->getStorage('friendship')->getQuery();

    $user_ids = [
      $this->currentUser->id(),
      $target_user->id(),
    ];

    // Get all relationships between this user.
    $query->condition('uid', $user_ids, 'IN')
      ->condition('requested_uid', $user_ids, 'IN')
      ->accessCheck(FALSE);

    return $query->execute();
  }

  /**
   * Get link attributes.
   *
   * @param \Drupal\user\Entity\User $target_user
   *   Target user.
   *
   * @return array
   *   Return link attributes.
   */
  public function getLinkAttributes(UserInterface $target_user): array {
    if ($this->isRequestSend($target_user)) {
      $link_attributes = [
        '#type' => 'link',
        '#title' => $this->config->get('button.unfollow_text'),
        '#url' => Url::fromRoute('friendship.unfollow', [
          'user' => $target_user->id(),
          'js' => 'nojs',
        ]),
      ];
    }
    elseif ($this->isFollowedYou($target_user)) {
      $link_attributes = [
        '#type' => 'link',
        '#title' => $this->config->get('button.accept_text'),
        '#url' => Url::fromRoute('friendship.accept', [
          'user' => $target_user->id(),
          'js' => 'nojs',
        ]),
      ];
    }
    elseif ($this->isFriend($target_user)) {
      $link_attributes = [
        '#type' => 'link',
        '#title' => $this->config->get('button.remove_friend_text'),
        '#url' => Url::fromRoute('friendship.removeFriend', [
          'user' => $target_user->id(),
          'js' => 'nojs',
        ]),
      ];
    }
    else {
      $link_attributes = [
        '#type' => 'link',
        '#title' => $this->config->get('button.follow_text'),
        '#url' => Url::fromRoute('friendship.follow', [
          'user' => $target_user->id(),
          'js' => 'nojs',
        ]),
      ];
    }

    return $link_attributes;
  }

  /**
   * Render friendship process link.
   *
   * @param \Drupal\user\Entity\User $target_user
   *   Target user.
   *
   * @return array
   *   Array with a link.
   */
  public function getProcessLink(UserInterface $target_user): array {
    $build = [];

    if (
      $target_user->id() != $this->currentUser->id() &&
      $this->currentUser->isAuthenticated()
    ) {
      $id_hash = md5($target_user->id() + rand());
      $build = [
        '#type' => 'link',
        '#attributes' => [
          'class' => [
            'use-ajax',
            'friendship-ajax-link-' . $target_user->id(),
          ],
          'id' => 'friendship-ajax-link-' . $id_hash,
        ],
        '#attached' => [
          'library' => [
            'core/drupal.ajax',
            'friendship/process-link',
          ],
        ],
        '#cache' => [
          'max-age' => 0,
        ],
      ];

      $link_attributes = $this->getLinkAttributes($target_user);
      $build += $link_attributes;
    }

    return $build;
  }

}
