<?php

namespace Drupal\frontend_editing_paragraphs\Hook;

use Drupal\Core\Entity\ContentEntityFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\Order;
use Drupal\Core\Render\Element;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\frontend_editing_paragraphs\Form\ParagraphAddForm;
use Drupal\frontend_editing_paragraphs\Form\ParagraphDeleteForm;
use Drupal\paragraphs\ParagraphInterface;

/**
 * Class for hooks from frontend_editing_paragraphs module.
 */
class FrontendEditingParagraphsHooks {

  use StringTranslationTrait;

  /**
   * Constructs hook class.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   */
  public function __construct(
    protected AccountProxyInterface $currentUser,
  ) {
  }

  /**
   * Implements hook_entity_type_build().
   */
  #[Hook('entity_type_build', order: Order::Last)]
  public function entityTypeBuild(array &$entity_types) {
    // Override the default delete form for paragraphs until the following issue
    // https://www.drupal.org/project/paragraphs_edit/issues/3343465 is fixed.
    $entity_types['paragraph']->setFormClass('entity_delete', ParagraphDeleteForm::class);
    $entity_types['paragraph']->setFormClass('entity_add', ParagraphAddForm::class);
  }

  /**
   * Implements hook_form_BASE_FORM_ID_alter() for paragraph_form.
   */
  #[Hook('form_paragraph_form_alter')]
  public function formParagraphFormAlter(&$form, FormStateInterface $form_state) {
    if (!$this->currentUser->hasPermission('edit behavior plugin settings')) {
      return;
    }

    /** @var \Drupal\Core\Entity\ContentEntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\paragraphs\ParagraphInterface $paragraphs_entity */
    $paragraphs_entity = $form_object->getEntity();

    // Ensure #parents is set.
    if (!isset($form['#parents'])) {
      $form['#parents'] = [];
    }

    // Build the behavior plugins fields, do not display behaviors when
    // translating and untranslatable fields are hidden.
    $paragraphs_type = $paragraphs_entity->getParagraphType();
    if ($paragraphs_type) {
      $form['behavior_plugins']['#tree'] = TRUE;
      $form['behavior_plugins']['#weight'] = -99;
      foreach ($paragraphs_type->getEnabledBehaviorPlugins() as $plugin_id => $plugin) {
        $plugin_definition = $plugin->getPluginDefinition();
        $form['behavior_plugins'][$plugin_id] = [
          '#type' => 'container',
          '#group' => implode('][', array_merge($form['#parents'], ['paragraph_behavior'])),
          '#parents' => array_merge($form['#parents'], ['behavior_plugins', $plugin_id]),
          '#weight' => $plugin_definition['weight'] ?? 0,
        ];
        $subform_state = SubformState::createForSubform($form['behavior_plugins'][$plugin_id], $form, $form_state);
        if ($plugin_form = $plugin->buildBehaviorForm($paragraphs_entity, $form['behavior_plugins'][$plugin_id], $subform_state)) {
          $form['behavior_plugins'][$plugin_id] = $plugin_form;
          // Add the paragraphs-behavior class, so that we are able to show
          // and hide behavior fields, depending on the active perspective.
          $form['behavior_plugins'][$plugin_id]['#attributes']['class'][] = 'paragraphs-behavior';
        }
      }
    }

    // Wrap behavior_plugins with initially closed details element.
    if (!empty($form['behavior_plugins'])) {
      $has_behaviors = FALSE;
      foreach (Element::getVisibleChildren($form['behavior_plugins']) as $paragraph_behavior) {
        $visible_children = Element::getVisibleChildren($form['behavior_plugins'][$paragraph_behavior]);
        if (!empty($visible_children)) {
          $has_behaviors = TRUE;
          break;
        }
      }
      if ($has_behaviors) {
        $form['behavior_plugins']['#type'] = 'details';
        $form['behavior_plugins']['#title'] = $this->t('Settings');
      }
    }
  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter', order: Order::Last)]
  public function formAlter(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\Core\Entity\ContentEntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    if ($form_object instanceof ContentEntityFormInterface) {
      /** @var \Drupal\paragraphs\ParagraphInterface $paragraphs_entity */
      $paragraphs_entity = $form_object->getEntity();
      if ($paragraphs_entity instanceof ParagraphInterface) {
        // In case the preview is available.
        if (!empty($form['actions']['preview'])) {
          if (!isset($form['actions']['preview']['#access']) || $form['actions']['preview']['#access']) {
            array_unshift($form['actions']['preview']['#submit'], [$this, 'formParagraphFormSubmit']);
          }
        }
        $form['#validate'][] = [$this, 'formParagraphFormValidate'];
        array_unshift($form['actions']['submit']['#submit'], [$this, 'formParagraphFormSubmit']);
      }
    }
  }

  /**
   * Validation handler for the paragraph entity form.
   *
   * @param array $form
   *   The complete form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The state of the form.
   */
  public function formParagraphFormValidate(array &$form, FormStateInterface $form_state) {
    if (!$this->currentUser->hasPermission('edit behavior plugin settings')) {
      return;
    }

    /** @var \Drupal\Core\Entity\ContentEntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\paragraphs\ParagraphInterface $paragraphs_entity */
    $entity = $form_object->getEntity();

    // Validate all enabled behavior plugins.
    $paragraphs_type = $entity->getParagraphType();
    foreach ($paragraphs_type->getEnabledBehaviorPlugins() as $plugin_id => $plugin_values) {
      if (!empty($form['behavior_plugins'][$plugin_id])) {
        $subform_state = SubformState::createForSubform($form['behavior_plugins'][$plugin_id], $form_state->getCompleteForm(), $form_state);
        $plugin_values->validateBehaviorForm($entity, $form['behavior_plugins'][$plugin_id], $subform_state);
      }
    }
  }

  /**
   * Submit handler for the paragraph entity form.
   *
   * @param array $form
   *   The complete form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The state of the form.
   */
  public function formParagraphFormSubmit(array &$form, FormStateInterface $form_state) {
    if (!$this->currentUser->hasPermission('edit behavior plugin settings')) {
      return;
    }

    /** @var \Drupal\Core\Entity\ContentEntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\paragraphs\ParagraphInterface $paragraphs_entity */
    $paragraphs_entity = $form_object->getEntity();
    $values = $form_state->getValues();

    if (isset($values['behavior_plugins'])) {
      // Submit all enabled behavior plugins.
      $paragraphs_type = $paragraphs_entity->getParagraphType();
      foreach ($paragraphs_type->getEnabledBehaviorPlugins() as $plugin_id => $plugin_values) {
        if (!isset($values['behavior_plugins'][$plugin_id])) {
          $values['behavior_plugins'][$plugin_id] = [];
        }
        $subform_state = SubformState::createForSubform($form['behavior_plugins'][$plugin_id], $form, $form_state);
        $plugin_values->submitBehaviorForm($paragraphs_entity, $values['behavior_plugins'][$plugin_id], $subform_state);
      }
    }
  }

  /**
   * Implements hook_field_widget_single_element_form_alter().
   */
  #[Hook('field_widget_single_element_form_alter')]
  public function fieldWidgetSingleElementFormAlter(array &$element, FormStateInterface $form_state, array $context) {
    if ($context['widget']->getPluginId() == 'paragraph_view_mode') {
      $form = $form_state->getFormObject();
      if (property_exists($form, 'rootParent') && !empty($element['value']['#ajax'])) {
        unset($element['value']['#ajax']);
      }
    }
  }

}
