<?php

namespace Drupal\frontend_editing\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\frontend_editing\Ajax\CloseSidePanelCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Existing entity reference add form.
 */
class EntityReferenceAddForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->moduleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'frontend_editing_entity_reference_add_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?FieldDefinitionInterface $field_definition = NULL) {
    if (!empty($field_definition)) {
      $settings = $field_definition->getSettings();
      if ($settings['target_type'] == 'media' && $this->moduleHandler->moduleExists('media_library_form_element')) {
        $form['entity'] = [
          '#type' => 'media_library',
          '#allowed_bundles' => $settings['handler_settings']['target_bundles'],
          '#cardinality' => 1,
          '#title' => $field_definition->getLabel(),
          '#description' => $field_definition->getDescription(),
        ];
      }
      else {
        $form['entity'] = [
          '#title' => $field_definition->getLabel(),
          '#description' => $field_definition->getDescription(),
          '#type' => 'entity_autocomplete',
          '#target_type' => $settings['target_type'],
          '#selection_handler' => $settings['handler'],
          '#selection_settings' => $settings['handler_settings'],
        ];
      }
      $form['actions'] = ['#type' => 'actions'];
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Add'),
        '#ajax' => [
          'callback' => '::ajaxSubmit',
          'progress' => [
            'type' => 'throbber',
            'message' => $this->t('Saving...'),
          ],
        ],
      ];
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $parent_entity_type = $this->getRouteMatch()->getParameter('parent_type');
    $parent_entity_id = $this->getRouteMatch()->getParameter('parent');
    $parent_field = $this->getRouteMatch()->getParameter('parent_field_name');
    $current_item = $this->getRouteMatch()->getParameter('current_item');
    $parent_entity = $this->entityTypeManager->getStorage($parent_entity_type)->load($parent_entity_id);
    if (!$parent_entity) {
      return;
    }
    $entity = $form_state->getValue('entity');
    $before = $this->getRouteMatch()->getParameter('before') ?? FALSE;
    if ($current_item) {
      $values = $parent_entity->get($parent_field)->getValue();
      $new_values = [];
      foreach ($values as $value) {
        if ($before && $value['target_id'] == $current_item) {
          $new_values[] = [
            'target_id' => $entity,
          ];
        }
        $new_values[] = $value;
        if (!$before && $value['target_id'] == $current_item) {
          $new_values[] = [
            'target_id' => $entity,
          ];
        }
      }
      $parent_entity->set($parent_field, $new_values);
    }
    else {
      $parent_entity->get($parent_field)->appendItem(['target_id' => $entity]);
    }
    $parent_entity->save();
  }

  /**
   * Ajax submit.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The response.
   */
  public function ajaxSubmit(array $form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $parent_entity_id = $this->getRouteMatch()->getParameter('parent');
    $parent_entity_type_id = $this->getRouteMatch()->getParameter('parent_type');
    $parent_field_name = $this->getRouteMatch()->getParameter('parent_field_name');
    $view_mode_id = $this->getRequest()->get('view_mode_id', 'default');
    // Selector is the same as for the field wrapper @see
    // frontend_editing_entity_view_alter.
    $selector = '[data-frontend-editing="' . $parent_entity_type_id . '--' . $parent_entity_id . '--' . $parent_field_name . '--' . $view_mode_id . '"]';
    // In case of success, close the sidebar.
    $response->addCommand(new CloseSidePanelCommand($selector, $parent_entity_id, $parent_entity_type_id, $parent_field_name));
    return $response;
  }

}
