<?php

namespace Drupal\frontend_editing\Hook;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\Order;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\frontend_editing\Ajax\CloseSidePanelCommand;
use Drupal\frontend_editing\Ajax\EntityPreviewCommand;
use Drupal\frontend_editing\Ajax\ScrollTopCommand;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Contains form alter hooks.
 */
class FormAlter {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * Constructs hook class.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route matcher.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected RouteMatchInterface $routeMatch,
    #[Autowire(service: 'request_stack')]
    protected RequestStack $requestStack,
    protected ModuleHandlerInterface $moduleHandler,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected MessengerInterface $messenger,
  ) {
  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter', order: Order::Last)]
  public function formAlter(&$form, FormStateInterface $form_state, $form_id) {
    $form_object = $form_state->getFormObject();
    if ($form_object instanceof EntityForm) {
      /** @var \Drupal\Core\Entity\EntityInterface $entity */
      $entity = $form_object->getEntity();
      $config = $this->configFactory->get('frontend_editing.settings');
      $supported_entity_types = $config->get('entity_types');
      // If a value is 0 (for disabled), a search will evaluate to TRUE, e.g.
      // "'key' == 0" evaluates to TRUE, therefore, we need to perform strict
      // search.
      if (!in_array($entity->getEntityTypeId(), array_keys($supported_entity_types), TRUE)) {
        return;
      }
      if ($entity instanceof ContentEntityInterface) {
        $frontend_editing_routes = [
          'frontend_editing.form',
          'frontend_editing.entity_reference_item_add',
        ];
        $current_route_name = $this->routeMatch->getRouteName();
        // Make the form submit with ajax.
        if (in_array($current_route_name, $frontend_editing_routes)) {
          $form['#attributes']['class'][] = 'frontend-editing-form';
          if ($current_route_name == 'frontend_editing.form') {
            $bundle = $entity->get($entity->getEntityType()
              ->getKey('bundle'))->entity;
            if (!in_array($entity->bundle(), $supported_entity_types[$entity->getEntityTypeId()], TRUE)) {
              return;
            }
            $form['#title'] = $this->t('Editing "@bundle_name"', [
              '@bundle_name' => $bundle ? $bundle->label() : $entity->getEntityType()->getLabel(),
            ]);
          }
          $form_state->set('view_mode_id', $this->requestStack->getCurrentRequest()->query->get('view_mode_id', 'default'));
          $form['#after_build'][] = [get_class($this), 'clearGin'];
          $form['actions']['submit']['#ajax'] = [
            'event' => 'click',
            'callback' => [$this, 'frontendEditingSuccess'],
            'progress' => [
              'type' => 'throbber',
              'message' => $this->t('Saving...'),
            ],
          ];
          if ($current_route_name == 'frontend_editing.entity_reference_item_add') {
            if ($entity->getEntityTypeId() !== 'paragraph') {
              $form['actions']['submit']['#submit'][] = [$this, 'frontendEditingEntityReferenceAdd'];
              $form['#attached']['library'][] = 'frontend_editing/forms_helper';
            }
          }
          // In case the preview is available.
          if (!empty($form['actions']['preview'])) {
            if (!isset($form['actions']['preview']['#access']) || $form['actions']['preview']['#access']) {
              // Add the preview library.
              $form['frontend_editing']['#attached']['library'][] = 'frontend_editing/preview';
              $form['actions']['preview']['#attributes']['class'][] = 'frontend-editing-preview-button';
              // Add the preview ajax callback to the preview button.
              $form['actions']['preview']['#ajax'] = [
                'callback' => [$this, 'frontendEditingPreview'],
                'disable-refocus' => $config->get('automatic_preview'),
                'progress' => [
                  'type' => 'throbber',
                  'message' => $this->t('Previewing...'),
                ],
              ];
              // Add the automatic preview checkbox.
              $form['actions']['automatic_preview'] = [
                '#type' => 'checkbox',
                '#title' => $this->t('Automatic preview'),
                '#default_value' => $config->get('automatic_preview'),
                '#weight' => 100,
                '#attributes' => [
                  'class' => [
                    'frontend-editing-automatic-preview',
                  ],
                ],
              ];
              if ($config->get('allow_detach')) {
                $form['actions']['detach_editor'] = [
                  '#type' => 'button',
                  '#value' => $this->t('Detach Editor'),
                  '#weight' => 100,
                  '#attributes' => [
                    'class' => [
                      'frontend-editing-detach-editor',
                    ],
                    'data-uuid' => $entity->uuid(),
                  ],
                  '#attached' => [
                    'library' => [
                      'frontend_editing/detach_editor',
                    ],
                  ],
                ];
                if (!$this->moduleHandler->moduleExists('preview')) {
                  $form['actions']['detach_editor_warning'] = [
                    '#type' => 'container',
                    '#weight' => 101,
                    '#markup' => '<div class="frontend-editing-detach-editor-warning">' . $this->t('To ensure data integrity during detach/re-attach of windows, please install <a href="https://www.drupal.org/project/all_entity_preview" target="_blank">"All entity preview"</a> module') . '</div>',
                  ];
                }
              }
            }
          }
        }
      }
    }
  }

  /**
   * Remove gin libraries from the form.
   *
   * As frontend editing forms are not displayed with toolbars and all special
   * elements that Gin provides, some js causes errors.
   */
  public static function clearGin(array $element, FormStateInterface $form_state) {
    if (!empty($element['#attached']['library'])) {
      $libraries_to_remove = [
        'gin/more_actions',
        'gin/top_bar',
      ];
      if (empty($element['advanced'])) {
        $libraries_to_remove[] = 'gin/sidebar';
      }
      foreach ($element['#attached']['library'] as $key => $library) {
        if (in_array($library, $libraries_to_remove)) {
          unset($element['#attached']['library'][$key]);
        }
      }
    }
    if (!empty($element['status']['#group'])) {
      unset($element['status']['#group']);
    }
    if (!empty($element['gin_sticky_actions'])) {
      unset($element['gin_sticky_actions']);
    }
    if (!empty($element['gin_sidebar'])) {
      unset($element['gin_sidebar']);
    }
    if (!empty($element['gin_sidebar_overlay'])) {
      unset($element['gin_sidebar_overlay']);
    }
    if (!empty($element['gin_move_focus_to_sticky_bar'])) {
      unset($element['gin_move_focus_to_sticky_bar']);
    }
    return $element;
  }

  /**
   * Sends a message that the user requests preview.
   */
  public function frontendEditingPreview(array &$form, FormStateInterface $form_state) {
    $response = $this->frontendEditingValidationErrors($form, $form_state, TRUE);
    if (!empty($response->getCommands())) {
      return $response;
    }
    $entity = $form_state->getFormObject()->getEntity();
    // Preview of new content is different from existing content.
    if ($entity->isNew()) {
      $view_mode_id = $form_state->get('view_mode_id');
      $selector = '.frontend-editing--placeholder';
      $response->addCommand(new EntityPreviewCommand($selector, $entity->uuid(), $view_mode_id));
    }
    else {
      $response->addCommand(new EntityPreviewCommand('[data-fe-preview-content="' . $entity->uuid() . '"]'));
    }
    return $response;
  }

  /**
   * Handles validation errors for the frontend editing form.
   *
   * @param array $form
   *   The original form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param bool $preview
   *   Whether it is a preview or not.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The response object.
   */
  protected function frontendEditingValidationErrors(array $form, FormStateInterface $form_state, $preview = FALSE) {
    $response = new AjaxResponse();
    $errors = $form_state->getErrors();
    if (!empty($errors)) {
      // Add all error messages to the sidebar.
      foreach ($errors as $error) {
        $command = new MessageCommand($error, NULL, ['type' => 'error']);
        $response->addCommand($command);
      }
      if ($preview) {
        // The error messages are already sent to the window with MessageCommand
        // so we need to remove them, so they are not appearing on other pages
        // when they are already irrelevant.
        $this->messenger->deleteByType('error');
        return $response;
      }
      // Update the form in the sidebar, so the invalid elements show errors.
      $html_command = new ReplaceCommand('form', $form);
      $response->addCommand($html_command);
      // Scroll to the top of the sidebar, so that the error messages are
      // visible.
      $scroll_command = new ScrollTopCommand('body');
      $response->addCommand($scroll_command);
    }
    return $response;
  }

  /**
   * Sends a message that the form was successfully submitted.
   */
  public function frontendEditingSuccess(array &$form, FormStateInterface $form_state) {
    $response = $this->frontendEditingValidationErrors($form, $form_state);
    if (!empty($response->getCommands())) {
      return $response;
    }
    $entity = $form_state->getFormObject()->getEntity();
    $parent_entity_id = $this->routeMatch->getParameter('parent');
    $parent_entity_type_id = $this->routeMatch->getParameter('parent_type');
    $parent_field_name = $this->routeMatch->getParameter('parent_field_name');
    $view_mode_id = $form_state->get('view_mode_id');
    // For now only paragraphs update supports no reload.
    if ($entity->getEntityTypeId() == 'paragraph') {
      $parent_entity_id = $entity->getParentEntity()->id();
      $parent_entity_type_id = $entity->getParentEntity()->getEntityTypeId();
      $parent_field_name = $entity->get('parent_field_name')->value;
    }
    $entityUuid = $entity->uuid();
    // Selector is the same as for the field wrapper @see
    // frontend_editing_entity_view_alter.
    $selector = '[data-frontend-editing="' . $parent_entity_type_id . '--' . $parent_entity_id . '--' . $parent_field_name . '--' . $view_mode_id . '"]';
    // In case of success, close the sidebar.
    $response->addCommand(new CloseSidePanelCommand($selector, $parent_entity_id, $parent_entity_type_id, $parent_field_name, $entityUuid));
    return $response;
  }

  /**
   * Add entity reference item.
   */
  public function frontendEditingEntityReferenceAdd(array &$form, FormStateInterface $form_state) {
    $entity = $form_state->getFormObject()->getEntity();
    $parent_entity_type = $this->routeMatch->getParameter('parent_type');
    $parent_entity_id = $this->routeMatch->getParameter('parent');
    $parent_field = $this->routeMatch->getParameter('parent_field_name');
    $current_item = $this->routeMatch->getParameter('current_item');
    $parent_entity = $this->entityTypeManager->getStorage($parent_entity_type)->load($parent_entity_id);
    if (!$parent_entity) {
      return;
    }
    $before = $this->routeMatch->getParameter('before') ?? FALSE;
    if ($current_item) {
      $values = $parent_entity->get($parent_field)->getValue();
      $new_values = [];
      foreach ($values as $value) {
        if ($before && $value['target_id'] == $current_item) {
          $new_values[] = [
            'target_id' => $entity->id(),
          ];
        }
        $new_values[] = $value;
        if (!$before && $value['target_id'] == $current_item) {
          $new_values[] = [
            'target_id' => $entity->id(),
          ];
        }
      }
      $parent_entity->set($parent_field, $new_values);
    }
    else {
      $parent_entity->get($parent_field)->appendItem($entity);
    }
    $parent_entity->save();
  }

}
