<?php

namespace Drupal\frontend_editing\Hook;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\Order;
use Drupal\Core\Render\ElementInfoManager;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\frontend_editing\ToolbarItem;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Hook implementations for frontend_editing.
 */
class FrontendEditingHooks {

  use StringTranslationTrait;

  /**
   * Constructs FrontendEditing hooks.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory object.
   * @param \Drupal\Core\Routing\AdminContext $adminContext
   *   The admin context.
   * @param \Drupal\Core\Render\ElementInfoManager $elementInfoManager
   *   The element info manager.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route matcher.
   */
  public function __construct(
    #[Autowire(service: 'current_user')]
    protected AccountProxyInterface $currentUser,
    #[Autowire(service: 'config.factory')]
    protected ConfigFactoryInterface $configFactory,
    #[Autowire(service: 'router.admin_context')]
    protected AdminContext $adminContext,
    #[Autowire(service: 'plugin.manager.element_info')]
    protected ElementInfoManager $elementInfoManager,
    #[Autowire(service: 'current_route_match')]
    protected RouteMatchInterface $routeMatch,
  ) {}

  /**
   * Implements hook_toolbar().
   */
  #[Hook('toolbar_alter', order: Order::Last)]
  public function toolbarAlter(array &$items): void {
    $route_name = $this->routeMatch->getRouteName();
    $frontend_editing_routes = [
      'frontend_editing.form',
      'frontend_editing.entity_reference_item_add',
      'frontend_editing.item_add_page',
    ];
    if (!empty($route_name) && in_array($route_name, $frontend_editing_routes)) {
      $items = [];
    }

    if ($this->configFactory->get('frontend_editing.settings')->get('hide_contextual_links')) {
      if (isset($items['contextual'])) {
        unset($items['contextual']);
      }
    }
  }

  /**
   * Implements hook_toolbar().
   */
  #[Hook('toolbar')]
  public function toolbar() {
    $items = [];
    $cache_metadata = new CacheableMetadata();
    $cache_metadata->addCacheContexts(['user.permissions']);
    if (!$this->currentUser->hasPermission('access frontend editing')) {
      $cache_metadata->applyTo($items);
      return $items;
    }

    $config = $this->configFactory->get('frontend_editing.settings');
    $cache_metadata->addCacheableDependency($config);
    $cache_metadata->addCacheContexts(['route']);
    $ui_toggle_enabled = $config->get('ui_toggle');
    if ($ui_toggle_enabled || $this->adminContext->isAdminRoute()) {
      $cache_metadata->applyTo($items);
      return $items;
    }

    $items['toggle'] = [
      '#type' => 'toolbar_item',
      'tab' => [
        '#lazy_builder' => [
          'frontend_editing.toolbar_item:renderToggle',
          [],
        ],
        '#create_placeholder' => TRUE,
        '#cache' => [
          'tags' => [
            'frontend_editing:toggle',
          ],
        ],
      ],
      '#wrapper_attributes' => [
        'class' => [
          'frontend-editing-toolbar-toggle',
        ],
      ],
      '#weight' => -11,
    ];
    // \Drupal\toolbar\Element\ToolbarItem::preRenderToolbarItem adds an
    // #attributes property to each toolbar item's tab child automatically.
    // Lazy builders don't support an #attributes property so we need to
    // add another render callback to remove the #attributes property. We start
    // by adding the defaults, and then we append our own pre render callback.
    $items['toggle'] += $this->elementInfoManager->getInfo('toolbar_item');
    $items['toggle']['#pre_render'][] = [ToolbarItem::class, 'removeTabAttributes'];
    $cache_metadata->applyTo($items);
    return $items;
  }

  /**
   * Implements hook_ajax_render_alter().
   */
  #[Hook('ajax_render_alter')]
  public function ajaxRenderAlter(array &$data) {
    $route_name = $this->routeMatch->getRouteName();
    if (!empty($route_name) && strpos($route_name, 'frontend_editing') !== FALSE) {
      foreach ($data as &$command) {
        if ($command['command'] == 'insert' && $command['method'] == 'prepend') {
          if (strpos((string) $command['data'], 'data-drupal-messages')) {
            $command['selector'] = '[data-drupal-messages], [data-drupal-messages-fallback]';
            $command['method'] = 'replaceWith';
          }
        }
      }
    }
  }

  /**
   * Implements hook_extra_field_display_info_alter().
   */
  #[Hook('extra_field_display_info_alter')]
  public function extraFieldDisplayInfoAlter(array &$info) {
    $enabled_entity_types = $this->configFactory->get('frontend_editing.settings')->get('entity_types');
    $frontend_editing_enabled_bundles = [];
    foreach ($enabled_entity_types as $entity_type => $bundles) {
      foreach ($bundles as $bundle) {
        $frontend_editing_enabled_bundles[] = $entity_type . '.' . $bundle;
      }
    }
    $info['frontend_editing']['bundles'] = $frontend_editing_enabled_bundles;
  }

  /**
   * Implements hook_gin_content_form_routes_alter().
   */
  #[Hook('gin_content_form_routes_alter')]
  public function ginContentFormRoutesAlter(array &$route_names) {
    // Allow Gin to process the form as it is a content form.
    $route_names[] = 'frontend_editing.form';
  }

  /**
   * Implements hook_theme_suggestions_page_alter().
   */
  #[Hook('theme_suggestions_page_alter', order: Order::Last)]
  public function themeSuggestionsPageAlter(array &$suggestions, array $variables) {
    if (in_array('page__frontend_editing', $suggestions)) {
      // Trick Gin to use page--frontend-editing.html.twig on Frontend Editing
      // pages as it checks for suggestion `page__node` in the array, and
      // doesn't add it again. But if the suggestion is the first and there is
      // still `page__frontend_editing` after it, the last will be used.
      array_unshift($suggestions, 'page__node');
    }
  }

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme($existing, $type, $theme, $path) {
    return [
      'html__frontend_editing' => [
        'template' => 'html--frontend-editing',
        'base hook' => 'html',
      ],
      'page__frontend_editing' => [
        'template' => 'page--frontend-editing',
        'base hook' => 'page',
      ],
    ];
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match) {
    switch ($route_name) {
      // Main module help for the frontend_editing module.
      case 'help.page.frontend_editing':
        $output = '<h3>' . $this->t('About') . '</h3>';
        $output .= '<p>' . $this->t('This module allows to edit entities directly in the
      frontend using a side panel.') . '</p>';
        $output .= '<p>' . $this->t('It is built as a lightweight integration, that does
      not rely on cores Settings Tray, Contextual Links, or Quick Edit
      modules') . '</p>';
        $output .= '<p>' . $this->t('Instead this module adds the link to editing forms of
      entities to the frontend, and provides a sidebar to load these forms as
      iframes.') . '</p>';
        $output .= '<p>' . $this->t('The benefit of this approach is, that editing in the
      frontend uses the same styles as editing in the backend. This allows a
      consistent editorial experience.') . '</p>';
        $output .= '<p>' . $this->t('The module works nicely with paragraphs and blocks,
      but other entities can be edited, too.') . '</p>';
        return $output;

      default:
        return '';
    }
  }

}
