<?php

namespace Drupal\frontend_editing\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FormatterInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Class ThirdPartySettings contains hooks for third party settings.
 */
class ThirdPartySettings {

  use StringTranslationTrait;

  /**
   * Constructs hook class.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * Implements hook_field_formatter_third_party_settings_form().
   */
  #[Hook('field_formatter_third_party_settings_form')]
  public function fieldFormatterThirdPartySettingsForm(FormatterInterface $plugin, FieldDefinitionInterface $field_definition, $view_mode, array $form, FormStateInterface $form_state) {
    $element = [];
    $field_types_to_process = [
      'entity_reference',
      'entity_reference_revisions',
    ];
    if (in_array($field_definition->getType(), $field_types_to_process)) {
      $supported_entity_types = $this->configFactory
        ->get('frontend_editing.settings')
        ->get('entity_types');
      // Check if the bundle, that field is attached to, is allowed for frontend
      // editing.
      $is_allowed = !empty($supported_entity_types[$field_definition->getSetting('target_type')]);
      if ($is_allowed !== FALSE) {
        $element['ajax_content_update'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Frontend Editing: Enable ajax content update for this field'),
          '#default_value' => $plugin->getThirdPartySetting('frontend_editing', 'ajax_content_update', FALSE),
        ];
        $element['move_up_down'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Frontend Editing: Enable move up and move down for this field'),
          '#default_value' => $plugin->getThirdPartySetting('frontend_editing', 'move_up_down', FALSE),
          '#description' => $this->t('This operation requires update permission of the host (parent) entity.'),
        ];
        $element['add_items'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Frontend Editing: Enable add items for this field'),
          '#default_value' => $plugin->getThirdPartySetting('frontend_editing', 'add_items', FALSE),
          '#description' => $this->t('This operation requires update permission of the host (parent) entity.'),
        ];
        $element['delete_items'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Frontend Editing: Enable delete items for this field'),
          '#default_value' => $plugin->getThirdPartySetting('frontend_editing', 'delete_items', FALSE),
          '#description' => $this->t('This operation requires update permission of the host (parent) entity.'),
        ];
      }
    }
    return $element;
  }

  /**
   * Implements hook_field_formatter_settings_summary_alter().
   */
  #[Hook('field_formatter_settings_summary_alter')]
  public function fieldFormatterSettingsSummaryAlter(&$summary, $context) {
    $allowed_types = [
      'entity_reference',
      'entity_reference_revisions',
    ];
    if (in_array($context['field_definition']->getType(), $allowed_types)) {
      $settings = $context['formatter']->getThirdPartySetting('frontend_editing', 'ajax_content_update', FALSE);
      if ($settings) {
        $summary[] = $this->t('Frontend Editing: Ajax content update enabled');
      }
      $settings = $context['formatter']->getThirdPartySetting('frontend_editing', 'move_up_down', FALSE);
      if ($settings) {
        $summary[] = $this->t('Frontend Editing: Move references up and down enabled');
      }
      $settings = $context['formatter']->getThirdPartySetting('frontend_editing', 'add_items', FALSE);
      if ($settings) {
        $summary[] = $this->t('Frontend Editing: Adding references enabled');
      }
      $settings = $context['formatter']->getThirdPartySetting('frontend_editing', 'delete_items', FALSE);
      if ($settings) {
        $summary[] = $this->t('Frontend Editing: Deleting references enabled');
      }
    }
  }

}
