/**
 * Implements frontend editing.
 */

(function (Drupal, once, window, document) {
  // Set global variable for dirty form, so all js files can use it.
  window.feFormDirty = false;

  // Onload event to scroll when whole document has loaded.
  window.addEventListener('load', (event) => {
    const sessionStorage = window.sessionStorage;
    const entityToScroll = sessionStorage.getItem('scroll_to');
    if (entityToScroll) {
      const elementToScroll = document.querySelector(
        `[data-uuid="${entityToScroll}"]`,
      );
      if (elementToScroll) {
        elementToScroll.scrollIntoView({
          behavior: 'smooth',
          block: 'center',
          inline: 'center',
        });
      }
      sessionStorage.removeItem('scroll_to');
    }
  });

  /**
   * Listen for postMessage events from the iframe.
   */
  window.addEventListener('message', function (event) {
    // Do we trust the sender of this message?
    if (event.origin !== window.location.origin) {
      return;
    }
    if (event.data === '' || event.data.length === 0) {
      return;
    }
    // Check that it is the message from frontend editing.
    if (typeof event.data.formUpdated === 'undefined') {
      return;
    }
    window.feFormDirty = true;
  });
  // Try to prevent page from losing the changes.
  window.addEventListener('beforeunload', async (event) => {
    if (window.feFormDirty) {
      event.preventDefault();
      event.returnValue = true;
      return Drupal.t(
        'Are you sure you want to close the window? All unsaved changes might be lost.',
      );
    }
  });

  // Global variables
  let sidebarWidth = 30;
  let sidebarFullWidth = 70;
  const editWideClass = 'editing-container--wide';

  Drupal.frontendEditing = Drupal.frontendEditing || {};

  Drupal.frontendEditing.sidebarResizer = function (editContainer) {
    const $sidebarResizer = document.createElement('div');
    $sidebarResizer.classList.add('sidebar-resizer');
    editContainer.append($sidebarResizer);

    let isResizing = false;
    let resizedWidth;

    $sidebarResizer.addEventListener('mousedown', (e) => {
      isResizing = true;

      document.body.style.cursor = 'ew-resize';
      document.body.style.userSelect = 'none';
      editContainer.style.transition = 'none';
    });

    window.addEventListener('mousemove', (e) => {
      if (!isResizing) return;

      // Prevent text selection and dragging.
      editContainer.style.pointerEvents = 'none';

      const windowWidth = window.innerWidth;
      const newWidth = windowWidth - e.clientX;
      const newWidthPercent = (newWidth / windowWidth) * 100;

      // Set a min and max width for the sidebar.
      // -5% from the default width is the minimum width of the sidebar.
      if (newWidthPercent >= sidebarWidth - 5 && newWidthPercent < 90) {
        editContainer.style.width = `${newWidthPercent}%`;
        resizedWidth = newWidthPercent;
      }
    });

    window.addEventListener('mouseup', () => {
      if (!isResizing) return;

      editContainer.style.pointerEvents = 'auto';
      document.body.style.cursor = '';
      document.body.style.userSelect = '';
      editContainer.style.transition = '';

      if (resizedWidth > sidebarWidth) {
        editContainer.classList.add(editWideClass);
      } else {
        editContainer.classList.remove(editWideClass);
      }

      isResizing = false;
    });
  };

  // Remove all placeholders.
  Drupal.frontendEditing.removePlaceholder = function () {
    // Delete all previews placeholders.
    const previewElements = document.querySelectorAll(
      '.frontend-editing--placeholder',
    );
    previewElements.forEach(function (previewElement) {
      previewElement.remove();
    });
    const activeEditingElements = document.querySelectorAll(
      '.frontend-editing--active-editing',
    );
    activeEditingElements.forEach(function (activeEditingElement) {
      activeEditingElement.classList.remove('frontend-editing--active-editing');
      activeEditingElement.classList.remove('frontend-editing--outline');
    });
  };

  // Callback for click function on an editable element.
  Drupal.frontendEditing.editingClick = function (e) {
    e.preventDefault();
    // Setup container.
    // Frontend-editing sidebar and full widths.
    const wideClassWidth = `${sidebarFullWidth}%`;
    const sidebarClassWidth = `${sidebarWidth}%`;

    let editContainer = document.getElementById('editing-container');
    if (!editContainer) {
      editContainer = document.createElement('div');
      editContainer.id = 'editing-container';
      editContainer.classList.add(
        'editing-container',
        'editing-container--loading',
      );

      document.body.append(editContainer);

      // Add sidebar resizer.
      Drupal.frontendEditing.sidebarResizer(editContainer);

      editContainer.style.width = sidebarClassWidth;
    } else {
      editContainer.innerHTML = '';
    }
    // Setup width toggle button.
    const widthToggle = document.createElement('button');
    widthToggle.type = 'button';
    widthToggle.className = 'editing-container__toggle';
    widthToggle.addEventListener('click', function (e) {
      const currentWidth = parseFloat(editContainer.style.width) || 0;
      if (
        editContainer.classList.contains(editWideClass) ||
        currentWidth > sidebarWidth
      ) {
        editContainer.classList.remove(editWideClass);
        editContainer.style.width = sidebarClassWidth;
      } else {
        editContainer.classList.add(editWideClass);
        editContainer.style.width = wideClassWidth;
      }
    });
    // Setup close button.
    const editClose = document.createElement('button');
    editClose.className = 'editing-container__close';
    editClose.type = 'button';
    editClose.addEventListener('click', async function (e) {
      if (window.feFormDirty) {
        const dialog = new ConfirmDialog({
          trueButtonText: Drupal.t('Yes'),
          falseButtonText: Drupal.t('No'),
          questionText: Drupal.t(
            'Are you sure you want to close the window? All unsaved changes might be lost.',
          ),
        });

        const shouldClose = await dialog.confirm();
        if (shouldClose) {
          window.feFormDirty = false;
          editContainer.remove();
          Drupal.frontendEditing.removePlaceholder();
        }
      } else {
        window.feFormDirty = false;
        editContainer.remove();
        Drupal.frontendEditing.removePlaceholder();
      }
    });

    const actions = document.createElement('div');
    actions.classList.add('editing-container__actions');
    actions.appendChild(widthToggle);
    actions.appendChild(editClose);

    // Populate container.
    editContainer.appendChild(actions);

    // Load editing iFrame and append.
    const iframe = document.createElement('iframe');
    iframe.onload = function () {
      editContainer.classList.remove('editing-container--loading');
    };
    editContainer.appendChild(iframe);
    iframe.src = e.target.href;

    Drupal.frontendEditing.removePlaceholder();
    const entityContainer = e.target.closest('.frontend-editing');
    if (e.target.classList.contains('frontend-editing__action--edit')) {
      entityContainer.classList.add('frontend-editing--active-editing');
    }
    // Add placeholder for preview if add actions were triggered.
    if (
      e.target.classList.contains('frontend-editing__action--after') ||
      e.target.classList.contains('frontend-editing__action--before')
    ) {
      if (entityContainer.dataset.preview) {
        const placeholder = document.createElement(entityContainer.tagName);
        placeholder.innerHTML = Drupal.t(
          'Here will be your new content. Click "Preview" button in the form to see it.',
        );
        placeholder.classList.add('frontend-editing');
        placeholder.classList.add('frontend-editing--placeholder');
        if (e.target.classList.contains('frontend-editing__action--after')) {
          placeholder.dataset.preview = `${entityContainer.dataset.preview}-after`;
          entityContainer.parentNode.insertBefore(
            placeholder,
            entityContainer.nextSibling,
          );
        }
        if (e.target.classList.contains('frontend-editing__action--before')) {
          placeholder.dataset.preview = `${entityContainer.dataset.preview}-before`;
          entityContainer.parentNode.insertBefore(
            placeholder,
            entityContainer.previousSibling,
          );
        }
      }
    }
  };

  /**
   * Ajax command feReloadPage.
   *
   * Reloads the page in case ajax content update failed.
   *
   * @param ajax
   * @param response
   * @param status
   */
  Drupal.AjaxCommands.prototype.feReloadPage = function (
    ajax,
    response,
    status,
  ) {
    if (status === 'success') {
      // Reload the page.
      window.location.reload();
    }
  };

  /**
   * Add callback for sidebar tray and add listeners to actions.
   */
  Drupal.behaviors.frontendEditing = {
    attach(context, settings) {
      sidebarFullWidth = settings.frontend_editing.full_width;
      sidebarWidth = settings.frontend_editing.sidebar_width;
      const actionsElements = once(
        'frontend-editing-processed',
        '.frontend-editing-actions',
        context,
      );
      // Find all elements with frontend editing action buttons and attach
      // event listener.
      actionsElements.forEach(function (actionsElement) {
        actionsElement.addEventListener('mouseover', function () {
          const container = actionsElement.closest('.frontend-editing');
          if (container) {
            container.classList.add('frontend-editing--outline');
          }
        });

        actionsElement.addEventListener('mouseout', function () {
          const container = actionsElement.closest('.frontend-editing');
          if (
            container &&
            !container.classList.contains('frontend-editing--active-editing')
          ) {
            container.classList.remove('frontend-editing--outline');
          }
        });

        actionsElement
          .querySelectorAll('.frontend-editing__action')
          .forEach(function (editingElement, i) {
            if (
              editingElement.classList.contains('frontend-editing-open-sidebar')
            ) {
              editingElement.addEventListener(
                'click',
                Drupal.frontendEditing.editingClick,
              );
            }
          });
      });
    },
  };
})(Drupal, once, window, window.document);
