/**
 * Implements preview.
 */
(function (Drupal, $, once) {
  /**
   * Implements automatic preview behaviour.
   */
  Drupal.behaviors.autoPreview = {
    attach(context, settings) {
      once(
        'autoPreview',
        '.frontend-editing-automatic-preview',
        context,
      ).forEach(function (element) {
        const form = element.closest('form');
        if (!form) {
          return;
        }
        const previewButton = form.querySelector(
          '.frontend-editing-preview-button',
        );
        const debounceTime = 300;

        // Listen to changes in the form and autocomplete fields.
        $(form).on(
          'formUpdated autocompleteclose',
          Drupal.debounce(function () {
            triggerPreview();
          }, debounceTime),
        );

        // Handle drop events in draggable tables.
        if (Drupal.tableDrag) {
          Drupal.tableDrag.prototype.onDrop = Drupal.debounce(function () {
            triggerPreview();
          }, debounceTime);
        }

        // Trigger preview if automatic preview is enabled.
        function triggerPreview() {
          if (element.checked) {
            // Prevent formUpdated behavior execution after triggering the preview.
            Drupal.behaviors.formUpdated.detach(context, settings, 'unload');
            if (typeof Drupal.file !== 'undefined') {
              $(previewButton).off('mousedown', Drupal.file.disableFields);
            }
            // Prevent preview button from taking the focus.
            previewButton.setAttribute('data-disable-refocus', true);
            previewButton.dispatchEvent(new Event('mousedown'));
          }
        }
      });
    },
  };

  /**
   * Ajax command feEntityPreview.
   *
   * @param ajax
   * @param response
   * @param status
   */
  Drupal.AjaxCommands.prototype.feEntityPreview = function (
    ajax,
    response,
    status,
  ) {
    if (status === 'success') {
      const messages = new Drupal.Message();
      messages.clear();
      let previewWindowOrigin = window;
      if (window.name === 'frontend-edit-detached') {
        previewWindowOrigin = window.opener;
      }

      if (!response.selector) {
        // Reload the page.
        previewWindowOrigin.parent.location.reload();
      } else {
        previewWindowOrigin.parent.postMessage(
          response,
          previewWindowOrigin.location.origin,
        );
      }
    }
  };
})(Drupal, jQuery, once);
