<?php

namespace Drupal\frontend_editing;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotAcceptableHttpException;

/**
 * Class FrontendEditingFormBuilder to prepare entity forms for FE.
 */
class FrontendEditingFormBuilder implements FrontendEditingFormBuilderInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity form builder.
   *
   * @var \Drupal\Core\Entity\EntityFormBuilderInterface
   */
  protected EntityFormBuilderInterface $entityFormBuilder;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected FormBuilderInterface $formBuilder;

  /**
   * Constructs FrontendEditingFormBuilder.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $entity_form_builder
   *   The entity form builder.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityFormBuilderInterface $entity_form_builder, FormBuilderInterface $form_builder) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFormBuilder = $entity_form_builder;
    $this->formBuilder = $form_builder;
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(ContentEntityInterface $entity, $operation = 'edit', $display = 'default') {
    $form_type = 'build' . ucfirst($operation) . 'Form';
    if (!method_exists($this, $form_type)) {
      throw new NotAcceptableHttpException("$operation is not supported by the FrontendEditing form builder.");
    }
    return $this->{$form_type}($entity, $operation, $display);
  }

  /**
   * Builds content entity edit form.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The given entity.
   * @param string $operation
   *   The operation.
   * @param string $display
   *   The form display name.
   * @param array $form_state_additions
   *   The form_state_additions.
   *
   * @return array
   *   The entity edit form.
   */
  public function buildEditForm(ContentEntityInterface $entity, $operation = 'edit', $display = 'default', $form_state_additions = []) {
    if (!$entity->access('update')) {
      throw new AccessDeniedHttpException();
    }
    $form_object = $this->entityTypeManager->getFormObject($entity->getEntityTypeId(), $operation);
    $form_object->setEntity($entity);
    $form_state = (new FormState())->setFormState($form_state_additions);
    // Flag that this form has been initialized.
    $form_state->set('entity_form_initialized', TRUE);
    $form_display = EntityFormDisplay::collectRenderDisplay($entity, $display);
    $form_object->setFormDisplay($form_display, $form_state);
    $entityForm = $this->formBuilder->buildForm($form_object, $form_state);
    if ($entity->hasLinkTemplate('edit-form')) {
      $entityForm['#action'] = $entity->toUrl('edit-form')->toString();
    }

    $delete_url = Url::fromRoute('frontend_editing.form', [
      'id' => $entity->id(),
      'type' => $entity->getEntityTypeId(),
      'operation' => 'delete',
    ]);
    $delete_access = $entity->isNew() || $entity->access('delete');
    $entityForm['actions']['delete'] = [
      '#type' => 'link',
      '#title' => $this->t('Delete'),
      '#url' => $delete_url,
      '#access' => $delete_access,
      '#attributes' => [
        'class' => ['button', 'button--danger'],
      ],
      '#weight' => 10,
    ];
    if ($entity->getEntityTypeId() !== 'node' && function_exists('_gin_form_actions')) {
      // Handle forms processed with Gin theme. This is needed because
      // delete button will not be shown on the form as actions are already
      // in static cache and will never be retrieved from the form render
      // array.
      $gin_actions = _gin_form_actions();
      if (!empty($gin_actions['actions'])) {
        // Remove the sticky class from actions, as in sidebar there is no
        // place for sticky.
        array_shift($entityForm['actions']['#attributes']['class']);
        $gin_actions['actions']['delete'] = $entityForm['actions']['delete'];
        if (isset($gin_actions['more_actions'])) {
          unset($gin_actions['more_actions']);
        }
        if (isset($gin_actions['gin_sidebar_toggle'])) {
          unset($gin_actions['gin_sidebar_toggle']);
        }
        _gin_form_actions($gin_actions);
      }
    }

    return $entityForm;
  }

  /**
   * Builds content entity delete form.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The given entity.
   * @param string $operation
   *   The operation.
   * @param string $display
   *   The form display name.
   * @param array $form_state_additions
   *   The form_state_additions.
   *
   * @return array
   *   The entity delete form.
   */
  public function buildDeleteForm(ContentEntityInterface $entity, $operation = 'delete', $display = 'default', $form_state_additions = []) {
    if (!$entity->access('delete')) {
      throw new AccessDeniedHttpException();
    }
    $entityForm = $this->entityFormBuilder->getForm($entity, $operation, $form_state_additions);
    $entityForm['title'] = [
      '#markup' => '<h3>' . $this->t('Are you sure you want to delete this @type?', ['@type' => $entity->getEntityType()->getSingularLabel()]) . '</h3>',
      '#weight' => -10,
    ];
    if ($entity->hasLinkTemplate('delete-form')) {
      // Update form action attribute with form delete uri.
      $entityForm['#action'] = $entity->toUrl('delete-form')->toString();
    }
    if (function_exists('_gin_form_actions')) {
      // Remove sticky class from form actions.
      array_shift($entityForm['actions']['#attributes']['class']);
    }
    return $entityForm;
  }

}
