<?php

namespace Drupal\frontend_editing;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Extension\ExtensionList;
use Drupal\Core\Render\Markup;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\frontend_editing\Event\EntityReferenceAddAccess;
use Drupal\frontend_editing\Event\EntityReferenceMoveAccess;
use Drupal\frontend_editing\Event\FrontendEditingEvents;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Helper class for field reference.
 */
class FieldReferenceHelper implements TrustedCallbackInterface, FieldReferenceHelperInterface {

  use StringTranslationTrait;

  /**
   * The config object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * The redirect destination.
   *
   * @var \Drupal\Core\Routing\RedirectDestinationInterface
   */
  protected $redirectDestination;

  /**
   * The module list.
   *
   * @var \Drupal\Core\Extension\ExtensionList
   */
  protected $extensionList;

  /**
   * Constructs a new FieldReferenceHelper object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $redirect_destination
   *   The redirect destination..
   * @param \Drupal\Core\Extension\ExtensionList $extension_list
   *   The module list.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, AccountProxyInterface $current_user, EntityRepositoryInterface $entity_repository, EventDispatcherInterface $event_dispatcher, RedirectDestinationInterface $redirect_destination, ExtensionList $extension_list) {
    $this->config = $config_factory->get('frontend_editing.settings');
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
    $this->entityRepository = $entity_repository;
    $this->eventDispatcher = $event_dispatcher;
    $this->redirectDestination = $redirect_destination;
    $this->extensionList = $extension_list;
  }

  /**
   * Prepares the build array.
   *
   * @param array $build
   *   The build array.
   *
   * @return array
   *   The build array.
   */
  protected function prepareBuild(array $build) {
    if (!isset($build['#attributes']['class'])) {
      $build['#attributes']['class'] = [];
    }
    if (!in_array('frontend-editing', $build['#attributes']['class'])) {
      $build['#attributes']['class'][] = 'frontend-editing';
    }
    if (!isset($build['frontend_editing'])) {
      $build['frontend_editing'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'frontend-editing-actions',
          ],
          'data-entity-type' => $build['#entity_type'],
        ],
        '#access' => $this->currentUser->hasPermission('access frontend editing'),
      ];
    }
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function accessAdd($parent_type, $parent, $parent_field_name) {
    $parent_entity = $this->entityTypeManager->getStorage($parent_type)->load($parent);
    return $this->allowAdd($parent_entity, $parent_field_name);
  }

  /**
   * {@inheritdoc}
   */
  public function accessAddType($bundle, $parent_type, $parent, $parent_field_name) {
    try {
      $parent_entity = $this->entityTypeManager->getStorage($parent_type)
        ->load($parent);
      $parent_entity = $this->entityRepository->getTranslationFromContext($parent_entity);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      return AccessResult::forbidden();
    }
    $add_access = $this->allowAdd($parent_entity, $parent_field_name);
    if (!$add_access->isAllowed()) {
      return $add_access;
    }
    // Check that the given bundle is allowed.
    $parent_field_definition = $parent_entity->get($parent_field_name)->getFieldDefinition();
    $handler_settings = $parent_field_definition->getSetting('handler_settings');
    if (!empty($handler_settings['target_bundles'])) {
      if (!empty($handler_settings['negate'])) {
        if (in_array($bundle, $handler_settings['target_bundles'])) {
          return AccessResult::forbidden();
        }
      }
      elseif (!in_array($bundle, $handler_settings['target_bundles'])) {
        return AccessResult::forbidden();
      }
    }
    $referenced_entity_type = $parent_field_definition->getSetting('target_type');
    // Check if user is allowed to create items of referenced type and bundle.
    $create_access = $this->entityTypeManager
      ->getAccessControlHandler($referenced_entity_type)
      ->createAccess($bundle, $this->currentUser, [], TRUE);
    if (!$create_access->isAllowed()) {
      return $create_access;
    }
    $event = new EntityReferenceAddAccess($parent_entity, $parent_field_name, $bundle);
    $this->eventDispatcher->dispatch($event, FrontendEditingEvents::FE_ENTITY_REFERENCE_ADD_ACCESS);
    return $event->getAccessResult();
  }

  /**
   * {@inheritdoc}
   */
  public function accessMove($parent_type, $parent, $parent_field_name, $current_item, $operation) {
    $parent_entity = $this->entityTypeManager->getStorage($parent_type)->load($parent);
    $parent_field_definition = $parent_entity->getFieldDefinition($parent_field_name);
    $entity = $this->entityTypeManager->getStorage($parent_field_definition->getSetting('target_type'))->load($current_item);
    return $this->allowMove($parent_entity, $parent_field_name, $entity, $operation);
  }

  /**
   * Checks if the user is allowed to add a new reference to entity field.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $parent_entity
   *   The parent entity.
   * @param string $parent_field_name
   *   The field name.
   *
   * @return bool|\Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  protected function allowAdd(FieldableEntityInterface $parent_entity, $parent_field_name) {
    if (!$this->currentUser->hasPermission('frontend editing add items')) {
      return AccessResult::forbidden('User does not have permission to add items.');
    }
    if (!$parent_entity->access('update') || !$parent_entity->hasField($parent_field_name)) {
      return AccessResult::forbidden();
    }
    $parent_field_definition = $parent_entity->get($parent_field_name)->getFieldDefinition();
    if ($parent_field_definition->getType() == 'entity_reference_revisions') {
      if ($parent_entity->isTranslatable() && !$parent_entity->isDefaultTranslation() && !$parent_field_definition->isTranslatable()) {
        return AccessResult::forbidden('The parent entity reference revisions field is not translatable.');
      }
    }
    $cardinality = $parent_field_definition->getFieldStorageDefinition()->getCardinality();
    if ($cardinality > 0 && $parent_entity->get($parent_field_name)->count() == $cardinality) {
      return AccessResult::forbidden('The parent entity reference field has reached its maximum cardinality.');
    }

    return AccessResult::allowed();
  }

  /**
   * Checks if the user is allowed to move a reference to entity field.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $parent_entity
   *   The parent entity.
   * @param string $reference_field_name
   *   The field name.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The current item ID.
   * @param string $operation
   *   The operation.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   The access result.
   */
  protected function allowMove(FieldableEntityInterface $parent_entity, $reference_field_name, EntityInterface $entity, $operation) {
    if (!$this->currentUser->hasPermission('frontend editing move items')) {
      return AccessResult::forbidden('User does not have permission to move items.');
    }
    // Check that the operation is valid.
    if (!in_array($operation, ['up', 'down'])) {
      return AccessResult::forbidden();
    }
    if ($entity->isNew()) {
      return AccessResult::forbidden();
    }
    // Load the parent entity translation according to context, if it exists.
    $parent = $this->entityRepository->getTranslationFromContext($parent_entity);
    // Check that the parent entity exists and the user has update access.
    if (!$parent || !$parent->access('update')) {
      return AccessResult::forbidden();
    }
    // Check that the parent entity has the entity reference field.
    if (!$parent->hasField($reference_field_name) || $parent->get($reference_field_name)->isEmpty()) {
      return AccessResult::forbidden();
    }
    if ($entity->getEntityTypeId() == 'paragraph') {
      // By default, paragraph reference fields do not support translations. For
      // this reason we need to check in case the parent entity is translatable
      // that current translation is the default one and that the paragraph
      // field supports translations. In other case do not allow to move the
      // paragraph, because it will break the sync translation and can change
      // the way the paragraphs are sorted for other translations.
      // There are modules that allow async translation of paragraphs. In this
      // case it will still be possible to do, because then the field definition
      // will identify that the field is translatable.
      if ($parent->isTranslatable() && !$parent->isDefaultTranslation() && !$parent->get($reference_field_name)->getFieldDefinition()->isTranslatable()) {
        return AccessResult::forbidden();
      }
    }
    // Get the paragraph items.
    $reference_items = $parent->get($reference_field_name)->getValue();
    if ($operation == 'up') {
      $item = reset($reference_items);
    }
    else {
      $item = end($reference_items);
    }
    if ($item['target_id'] == $entity->id()) {
      return AccessResult::forbidden();
    }
    // Allow other modules to react to the move operation only if all other
    // checks passed.
    $event = new EntityReferenceMoveAccess($parent_entity, $reference_field_name, $entity, $operation);
    $this->eventDispatcher->dispatch($event, FrontendEditingEvents::FE_ENTITY_REFERENCE_MOVE_ACCESS);
    return $event->getAccessResult();
  }

  /**
   * {@inheritdoc}
   */
  public function addAddItemButtons(array $build) {
    $entity = $build['#' . $build['#entity_type']];
    $build = $this->prepareBuild($build);
    if (!empty($entity->_referringItem)) {
      $parent_entity = $entity->_referringItem->getEntity();
      $reference_field_name = $entity->_referringItem->getParent()->getName();
      $build['frontend_editing']['add_items'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'add-items ' . ($this->config->get('hover_highlight') ? ' hover-highlight' : ''),
          ],
        ],
        '#access' => $this->allowAdd($parent_entity, $reference_field_name)->isAllowed(),
      ];
      $path_to_plus_svg = $this->extensionList->getPath('frontend_editing') . '/images/svg/plus_circle.svg';
      $svg_markup = Markup::create(file_get_contents($path_to_plus_svg) ?? '');

      $build['frontend_editing']['add_items']['before'] = [
        '#type' => 'link',
        '#title' => $svg_markup,
        '#url' => Url::fromRoute('frontend_editing.item_add_page', [
          'parent_type' => $parent_entity->getEntityTypeId(),
          'parent' => $parent_entity->id(),
          'parent_field_name' => $reference_field_name,
          'current_item' => $entity->id(),
          'before' => 1,
        ], [
          'query' => [
            'view_mode_id' => $build['#parent_field_view_mode'],
          ],
        ]),
        '#attributes' => [
          'title' => $this->t('Add before'),
          'class' => [
            'frontend-editing-open-sidebar',
            'frontend-editing__action',
            'frontend-editing__action--before',
          ],
        ],
      ];
      $build['frontend_editing']['add_items']['after'] = [
        '#type' => 'link',
        '#title' => $svg_markup,
        '#url' => Url::fromRoute('frontend_editing.item_add_page', [
          'parent_type' => $parent_entity->getEntityTypeId(),
          'parent' => $parent_entity->id(),
          'parent_field_name' => $reference_field_name,
          'current_item' => $entity->id(),
          'before' => 0,
        ], [
          'query' => [
            'view_mode_id' => $build['#parent_field_view_mode'],
          ],
        ]),
        '#attributes' => [
          'title' => $this->t('Add after'),
          'class' => [
            'frontend-editing-open-sidebar',
            'frontend-editing__action',
            'frontend-editing__action--after',
          ],
        ],
      ];
    }
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function addMoveButtons(array $build) {
    $entity = $build['#' . $build['#entity_type']];
    $build = $this->prepareBuild($build);
    if (!empty($entity->_referringItem)) {
      $parent_entity = $entity->_referringItem->getEntity();
      $reference_field_name = $entity->_referringItem->getParent()->getName();
      if (!isset($build['frontend_editing']['common_actions'])) {
        $build['frontend_editing']['common_actions'] = [
          '#type' => 'container',
          '#attributes' => [
            'class' => [
              'common-actions-container',
            ],
          ],
        ];
      }
      $entity = $build['#' . $build['#entity_type']];
      // Add the move up/down.
      $build['frontend_editing']['common_actions']['move_up_down'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'move-up-down icons-container',
          ],
        ],
      ];
      $cacheable_metadata = CacheableMetadata::createFromObject($entity->_referringItem);
      $cacheable_metadata->applyTo($build['frontend_editing']['common_actions']['move_up_down']);
      if ($this->allowMove($parent_entity, $reference_field_name, $entity, 'up')->isAllowed()) {
        $build['frontend_editing']['common_actions']['move_up_down']['up'] = [
          '#type' => 'link',
          '#title' => $this->t('Move up'),
          '#url' => Url::fromRoute('frontend_editing.item_move_up', [
            'current_item' => $entity->id(),
            'parent_type' => $parent_entity->getEntityTypeId(),
            'parent' => $parent_entity->id(),
            'parent_field_name' => $reference_field_name,
          ],
          [
            'query' => [
              'view_mode_id' => $build['#parent_field_view_mode'],
            ] + $this->redirectDestination->getAsArray(),
          ]),
          '#attributes' => [
            'title' => $this->t('Move up'),
            'class' => [
              'frontend-editing__action',
              'frontend-editing__action--up',
            ],
          ],
        ];
      }
      if ($this->allowMove($parent_entity, $reference_field_name, $entity, 'down')->isAllowed()) {
        $build['frontend_editing']['common_actions']['move_up_down']['down'] = [
          '#type' => 'link',
          '#title' => $this->t('Move down'),
          '#url' => Url::fromRoute('frontend_editing.item_move_down', [
            'current_item' => $entity->id(),
            'parent_type' => $parent_entity->getEntityTypeId(),
            'parent' => $parent_entity->id(),
            'parent_field_name' => $reference_field_name,
          ],
          [
            'query' => [
              'view_mode_id' => $build['#parent_field_view_mode'],
            ] + $this->redirectDestination->getAsArray(),
          ]),
          '#attributes' => [
            'title' => $this->t('Move down'),
            'class' => [
              'frontend-editing__action',
              'frontend-editing__action--down',
            ],
          ],
        ];
      }
    }
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function addAjaxContentUpdate(array $build) {
    if (!empty($build['frontend_editing']['common_actions']['move_up_down']['down'])) {
      $build['frontend_editing']['common_actions']['move_up_down']['down']['#attributes']['class'][] = 'use-ajax';
    }
    if (!empty($build['frontend_editing']['common_actions']['move_up_down']['up'])) {
      $build['frontend_editing']['common_actions']['move_up_down']['up']['#attributes']['class'][] = 'use-ajax';
    }
    if (!empty($build['frontend_editing']['common_actions']['title_edit']['edit'])) {
      /** @var \Drupal\Core\Url $url */
      $url = $build['frontend_editing']['common_actions']['title_edit']['edit']['#url'];
      $url->mergeOptions(['query' => ['view_mode_id' => $build['#parent_field_view_mode']]]);
      $build['frontend_editing']['common_actions']['title_edit']['edit']['#url'] = $url;
    }
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function move($parent_type, $parent, $parent_field_name, $current_item, $operation) {
    $parent_entity = $this->entityTypeManager->getStorage($parent_type)->load($parent);
    $parent_field_definition = $parent_entity->getFieldDefinition($parent_field_name);
    $entity = $this->entityTypeManager->getStorage($parent_field_definition->getSetting('target_type'))->load($current_item);
    $items = $parent_entity->get($parent_field_name)->getValue();
    if ($operation == 'up') {
      foreach ($items as $delta => $item) {
        if ($item['target_id'] == $entity->id()) {
          if ($delta > 0) {
            $prev_item = $items[$delta - 1];
            $items[$delta - 1] = $items[$delta];
            $items[$delta] = $prev_item;
          }
          break;
        }
      }
    }
    else {
      $numitems = count($items);
      foreach ($items as $delta => $item) {
        if ($item['target_id'] == $entity->id()) {
          if ($delta < $numitems) {
            $next_item = $items[$delta + 1];
            $items[$delta + 1] = $items[$delta];
            $items[$delta] = $next_item;
          }
          break;
        }
      }
    }
    $parent_entity->get($parent_field_name)->setValue($items);
    static::saveParentEntity($parent_entity, $entity);
    return TRUE;
  }

  /**
   * Saves parent entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $parent_entity
   *   The parent entity.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity from reference field.
   */
  public function saveParentEntity(ContentEntityInterface $parent_entity, ContentEntityInterface $entity) {
    $parent_entity->save();
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['addAddItemButtons', 'addMoveButtons', 'addAjaxContentUpdate'];
  }

}
