<?php

/**
 * @file
 * Post update hooks for the frontend_editing module.
 */

use Drupal\Core\Config\Entity\ConfigEntityUpdater;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;

/**
 * Grant 'add paragraphs' and 'move paragraphs' permissions.
 */
function frontend_editing_post_update_add_paragraph_permissions(&$sandbox) {
  // Get all roles.
  $roles = \Drupal::entityTypeManager()->getStorage('user_role')->loadMultiple();
  /** @var \Drupal\user\RoleInterface $role */
  foreach ($roles as $role) {
    // If role has permission to access frontend editing, allow to add and move
    // paragraphs for backwards compatibility.
    if ($role->hasPermission('access frontend editing')) {
      $role->grantPermission('add paragraphs');
      $role->grantPermission('move paragraphs');
      $role->save();
    }
  }
}

/**
 * Replace paragraph specific permissions with more generic ones.
 */
function frontend_editing_post_update_11001(&$sandbox) {
  // Get all roles.
  $roles = \Drupal::entityTypeManager()->getStorage('user_role')->loadMultiple();
  /** @var \Drupal\user\RoleInterface $role */
  foreach ($roles as $role) {
    $needs_save = FALSE;
    if ($role->hasPermission('add paragraphs')) {
      $role->grantPermission('frontend editing add items');
      $role->revokePermission('add paragraphs');
      $needs_save = TRUE;
    }
    if ($role->hasPermission('move paragraphs')) {
      $role->grantPermission('frontend editing move items');
      $role->revokePermission('move paragraphs');
      $needs_save = TRUE;
    }
    if ($needs_save) {
      $role->save();
    }
  }
}

/**
 * Enable frontend editing for previously selected bundles.
 */
function frontend_editing_post_update_11002() {
  // Clear the plugin manager cache to make sure the new definitions are loaded.
  \Drupal::service('plugin.manager.extra_field_display')->clearCache();
  \Drupal::service('plugin.manager.extra_field_display')->clearCachedDefinitions();
  // This is needed until this issue is resolved
  // https://www.drupal.org/project/extra_field_plus/issues/3436799 .
  \Drupal::service('plugin.manager.extra_field_plus_display')->clearCache();
  \Drupal::service('plugin.manager.extra_field_plus_display')->clearCachedDefinitions();
  // Clear extra fields cache definitions.
  \Drupal::service('entity_field.manager')->clearCachedFieldDefinitions();
}

/**
 * Update entity view displays to include frontend editing settings.
 */
function frontend_editing_post_update_11003(&$sandbox) {
  $config_entity_updater = \Drupal::classResolver(ConfigEntityUpdater::class);
  $callback = function (EntityViewDisplayInterface $entity_view_display) {
    $return = FALSE;
    $fields_exclude = \Drupal::config('frontend_editing.settings')->get('exclude_fields');
    $ajax_content_update = \Drupal::config('frontend_editing.settings')->get('ajax_content_update');
    foreach ($entity_view_display->getComponents() as $name => $component) {
      if (!isset($component['type'])) {
        continue;
      }
      if ($component['type'] == 'entity_reference_revisions_entity_view') {
        $component['third_party_settings']['frontend_editing'] = [];
        if ($ajax_content_update) {
          $field_machine_name = $entity_view_display->getTargetEntityTypeId() . '.' . $entity_view_display->getTargetBundle() . '.' . $name;
          if (in_array($field_machine_name, $fields_exclude)) {
            $ajax_content_update = FALSE;
          }
        }
        $component['third_party_settings']['frontend_editing']['ajax_content_update'] = $ajax_content_update;
        $component['third_party_settings']['frontend_editing']['move_up_down'] = TRUE;
        $component['third_party_settings']['frontend_editing']['add_items'] = TRUE;
        $entity_view_display->setComponent($name, $component);
        $return = TRUE;
      }
    }
    return $return;
  };
  $config_entity_updater->update($sandbox, 'entity_view_display', $callback);
}

/**
 * Clean up config after removing deprecated settings.
 */
function frontend_editing_post_update_11004() {
  $raw_data = \Drupal::configFactory()->getEditable('frontend_editing.settings')->getRawData();
  unset($raw_data['ajax_content_update']);
  unset($raw_data['exclude_fields']);
  \Drupal::configFactory()->getEditable('frontend_editing.settings')->setData($raw_data)->save(TRUE);
}
