<?php

namespace Drupal\frontend_editing\Plugin\ExtraField\Display;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\extra_field_plus\Plugin\ExtraFieldPlusDisplayBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Frontend Editing actions.
 *
 * @ExtraFieldDisplay(
 *   id = "frontend_editing",
 *   label = @Translation("Frontend Editing: Actions"),
 *   bundles = {},
 *   visible = true
 * )
 */
class FrontendEditing extends ExtraFieldPlusDisplayBase implements ContainerFactoryPluginInterface {

  /**
   * The config object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The preview config object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $previewConfig;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new FrontendEditing object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory, AccountProxyInterface $current_user, ModuleHandlerInterface $module_handler) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->config = $config_factory->get('frontend_editing.settings');
    $this->previewConfig = $config_factory->get('preview.settings');
    $this->currentUser = $current_user;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('current_user'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function view(ContentEntityInterface $entity) {
    $settings = $this->getEntityExtraFieldSettings();
    $form_display = $settings['form_display'] ?? 'default';
    $element = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'frontend-editing-actions',
        ],
        'data-entity-type' => $entity->getEntityTypeId(),
        'data-uuid' => $entity->uuid(),
      ],
      '#access' => $this->currentUser->hasPermission('access frontend editing') && $entity->access('update') && !($entity instanceof RevisionableInterface && !$entity->isLatestRevision()),
    ];
    // Attach necessary libraries and drupal settings to element.
    // Main library for sidebar editing.
    $element['#attached']['library'][] = 'frontend_editing/frontend_editing';
    // Library for updating content.and listen to the messages from frontend
    // editing iframe that is in the sidebar.
    $element['#attached']['library'][] = 'frontend_editing/update_content';
    // Add sidebar size configuration to drupal settings.
    $element['#attached']['drupalSettings']['frontend_editing'] = [
      'sidebar_width' => $this->config->get('sidebar_width') ?? 30,
      'full_width' => $this->config->get('full_width') ?? 70,
    ];
    if ($this->config->get('action_links_in_viewport') == 'floating') {
      // Library for keeping editing actions in viewport.
      $element['#attached']['library'][] = 'frontend_editing/floating_actions';
    }
    elseif ($this->config->get('action_links_in_viewport') == 'duplicate') {
      // Library for duplicating editing actions and add to the bottom of the
      // element.
      $element['#attached']['drupalSettings']['frontend_editing']['duplicate_action_links_height'] = $this->config->get('duplicate_action_links.height') ?? 300;
      $element['#attached']['library'][] = 'frontend_editing/duplicate_links';
    }

    $element['common_actions'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'common-actions-container',
        ],
      ],
    ];
    $element['common_actions']['title_edit'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'title-edit-container',
        ],
      ],
    ];
    if ($entity->getEntityType()->hasKey('bundle')) {
      $bundle_label = $entity->getEntityType()->getLabel();
      $bundle = $entity->get($entity->getEntityType()->getKey('bundle'))->entity;
      if ($bundle) {
        $bundle_label = $bundle->label();
      }
      $element['common_actions']['title_edit']['title'] = [
        '#markup' => '<span class="action-title">' . $bundle_label . '</span>',
      ];
    }
    // Add the edit link.
    $element['common_actions']['title_edit']['edit'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit'),
      '#url' => Url::fromRoute('frontend_editing.form', [
        'type' => $entity->getEntityTypeId(),
        'id' => $entity->id(),
        'display' => $form_display,
      ], [
        // Add uuid query parameter for entity preview.
        'query' => [
          'uuid' => $entity->uuid(),
        ],
      ]),
      '#attributes' => [
        'title' => $this->t('Edit'),
        'class' => [
          'frontend-editing-open-sidebar',
          'frontend-editing__action',
          'frontend-editing__action--edit',
        ],
      ],
    ];
    // Check whether it is possible to have preview of the changes.
    if ($this->moduleHandler->moduleExists('preview')) {
      // Check if preview is allowed.
      $preview_config = $this->previewConfig->get('enabled') ?? [];
      if (!empty($preview_config[$entity->getEntityTypeId()][$entity->bundle()]) || $entity->getEntityTypeId() == 'node') {
        $preview_url = Url::fromRoute('preview.entity_preview', [
          'entity_preview' => $entity->uuid(),
          'view_mode_id' => $this->getViewMode(),
        ]);
        if ($entity->getEntityTypeId() == 'node') {
          $preview_url = Url::fromRoute('entity.node.preview', [
            'node_preview' => $entity->uuid(),
            'view_mode_id' => $this->getViewMode(),
          ]);
        }
        $element['entity_preview'] = [
          '#type' => 'link',
          '#title' => $this->t('Preview entity'),
          '#url' => $preview_url,
          '#attributes' => [
            'title' => $this->t('Reload content with preview'),
            'class' => [
              'use-ajax',
              'frontend-editing-preview-content',
            ],
            'data-fe-preview-content' => $entity->uuid(),
          ],
        ];
      }
    }
    // Add cacheable metadata to rendered array.
    $cacheable_metadata = CacheableMetadata::createFromRenderArray($element);
    // In case the settings are changed and new entity types are added or some
    // are removed, we need to invalidate the cache.
    $cacheable_metadata->addCacheableDependency($this->config);
    // Also the user permissions are relevant for the rendered array.
    $cacheable_metadata->addCacheContexts(['user.permissions']);
    $cacheable_metadata->applyTo($element);
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function getExtraFieldSettingsForm(string $field_id, string $entity_type_id, string $bundle, string $view_mode = 'default'): array {
    $elements = parent::getExtraFieldSettingsForm($field_id, $entity_type_id, $bundle, $view_mode);
    /** @var \Drupal\Core\Entity\Display\EntityFormDisplayInterface[] $form_displays */
    $form_displays = \Drupal::entityTypeManager()->getStorage('entity_form_display')->loadByProperties([
      'targetEntityType' => $entity_type_id,
      'bundle' => $bundle,
      'status' => TRUE,
    ]);
    foreach ($form_displays as $form_display) {
      $elements['form_display']['#options'][$form_display->getMode()] = $form_display->getMode();
    }
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  protected static function extraFieldSettingsForm(): array {
    $form = parent::extraFieldSettingsForm();
    $options = [
      '' => t('- Select -'),
    ];
    $form['form_display'] = [
      '#type' => 'select',
      '#title' => t('Form display'),
      '#options' => $options,
      '#require' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  protected static function defaultExtraFieldSettings(): array {
    $values = parent::defaultExtraFieldSettings();
    $values += [
      'form_display' => 'default',
    ];
    return $values;
  }

  /**
   * {@inheritdoc}
   */
  protected static function settingsSummary(string $field_id, string $entity_type_id, string $bundle, string $view_mode = 'default'): array {
    return [
      t('Form display: @form_display', [
        '@form_display' => self::getExtraFieldSetting($field_id, 'form_display', $entity_type_id, $bundle, $view_mode),
      ]),
    ];
  }

}
