<?php

namespace Drupal\frontend_editing\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Render\Element;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\frontend_editing\FieldReferenceHelperInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Contains hooks that change entity display.
 */
class ViewAlter {

  use StringTranslationTrait;

  /**
   * Constructs hook class for view alter.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\frontend_editing\FieldReferenceHelperInterface $fieldReferenceHelper
   *   The field reference helper.
   * @param \Drupal\Core\Routing\AdminContext $adminContext
   *   The admin context.
   */
  public function __construct(
    #[Autowire(service: 'current_user')]
    protected AccountProxyInterface $currentUser,
    #[Autowire(service: 'config.factory')]
    protected ConfigFactoryInterface $configFactory,
    #[Autowire(service: 'module_handler')]
    protected ModuleHandlerInterface $moduleHandler,
    #[Autowire(service: 'frontend_editing.field_reference')]
    protected FieldReferenceHelperInterface $fieldReferenceHelper,
    #[Autowire(service: 'router.admin_context')]
    protected AdminContext $adminContext,
  ) {
  }

  /**
   * Implements hook_entity_view_alter().
   */
  #[Hook('entity_view_alter')]
  public function viewAlter(&$build, EntityInterface $entity, EntityViewDisplayInterface $display) {
    // No need to use frontend editing in admin UI.
    if ($this->adminContext->isAdminRoute()) {
      return;
    }
    // Don't do this for embedded entities.
    if (isset($build['#embed'])) {
      return;
    }
    $build['#cache']['contexts'][] = 'user.permissions';
    // Only do this, if user has access to frontend editing.
    if (!$this->currentUser->hasPermission('access frontend editing')) {
      return;
    }
    if ($this->moduleHandler->moduleExists('preview')) {
      // Check if preview is allowed.
      $preview_config = $this->configFactory->get('preview.settings')->get('enabled') ?? [];
      if (!empty($preview_config[$entity->getEntityTypeId()][$entity->bundle()]) || $entity->getEntityTypeId() == 'node') {
        $build['#attributes']['data-preview'] = $entity->uuid();
      }
    }
    // Don't do this for previews of not-yet-existing entities.
    if ($entity->isNew()) {
      $build['#attributes']['class'][] = 'frontend-editing--placeholder';
      return;
    }
    if (!empty($build['extra_field_frontend_editing'])) {
      $build['#attributes']['class'][] = 'frontend-editing';
      $build['frontend_editing'] = $build['extra_field_frontend_editing'];
      unset($build['extra_field_frontend_editing']);
    }
    else {
      $build['frontend_editing'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'frontend-editing-actions',
          ],
          'data-entity-type' => $entity->getEntityTypeId(),
        ],
        '#access' => $this->currentUser->hasPermission('access frontend editing') && $entity->access('update') && !($entity instanceof RevisionableInterface && !$entity->isLatestRevision()),
      ];
    }
    $frontend_editing_config = $this->configFactory->get('frontend_editing.settings');
    foreach (Element::getVisibleChildren($build) as $field_name) {
      $field_name_component = $display->getComponent($field_name);
      // Check if the field is empty.
      if (empty($build[$field_name]['#theme']) && !empty($field_name_component['third_party_settings']['frontend_editing']['add_items'])) {
        /** @var \Drupal\Core\Field\FieldDefinitionInterface $field_definition */
        $field_definition = $entity->getFieldDefinition($field_name);

        // Attach necessary libraries and drupal settings to FE element.
        // Main library for sidebar editing.
        $build[$field_name]['#attached']['library'][] = 'frontend_editing/frontend_editing';
        // Library for updating content.and listen to the messages from frontend
        // editing iframe that is in the sidebar.
        $build[$field_name]['#attached']['library'][] = 'frontend_editing/update_content';
        // Add sidebar size configuration to drupal settings.
        $build[$field_name]['#attached']['drupalSettings']['frontend_editing'] = [
          'sidebar_width' => $frontend_editing_config->get('sidebar_width') ?? 30,
          'full_width' => $frontend_editing_config->get('full_width') ?? 70,
        ];
        $build[$field_name]['#theme_wrappers']['container']['#attributes']['class'][] = 'frontend-editing-actions';
        $build[$field_name]['add_item'] = [
          '#type' => 'link',
          '#title' => $this->t('No items found for field "@label", click to add one.', ['@label' => $field_definition->getLabel()]),
          '#url' => Url::fromRoute('frontend_editing.item_add_page', [
            'parent_type' => $entity->getEntityTypeId(),
            'parent' => $entity->id(),
            'parent_field_name' => $field_name,
          ], [
            'query' => [
              'view_mode_id' => $build['#view_mode'] ?? 'default',
            ],
          ]),
          '#attributes' => [
            'title' => $this->t('Add'),
            'class' => [
              'frontend-editing-open-sidebar',
              'frontend-editing__action',
              'frontend-editing__action--add-empty',
            ],
          ],
          '#access' => $this->currentUser->hasPermission('access frontend editing'),
        ];
      }

      if (!isset($build[$field_name]['#theme'])) {
        continue;
      }
      if ($build[$field_name]['#theme'] != 'field') {
        continue;
      }
      // Make sure that third party formatter settings are valid for given
      // fields.
      $this->checkThirdPartyFormatterSettings($build[$field_name]);
      $ajax_content_update = FALSE;
      if (!empty($build[$field_name]['#third_party_settings']) && !empty($build[$field_name]['#third_party_settings']['frontend_editing']['ajax_content_update'])) {
        $ajax_content_update = TRUE;
      }
      if ($ajax_content_update) {
        // Selector should contain the entity type, the entity id and the field
        // name that it belongs too.
        $selector = $entity->getEntityTypeId() . '--' . $entity->id() . '--' . $field_name . '--' . $build[$field_name]['#view_mode'];
        // Because it is possible to remove wrappers in the field template and
        // there is no reliable way to know whether the field wrapper attributes
        // are printed, force add theme wrappers.
        $build[$field_name]['#theme_wrappers'] = [
          'container' => [
            '#attributes' => [
              'data-frontend-editing' => $selector,
              'class' => [
                'frontend-editing-field-wrapper',
              ],
            ],
          ],
        ];
        $build['frontend_editing'][$field_name . '_update_content'] = [
          '#type' => 'link',
          '#title' => $this->t('Reload content for this field'),
          '#url' => Url::fromRoute('frontend_editing.update_content', [
            'entity_id' => $entity->id(),
            'entity_type_id' => $entity->getEntityTypeId(),
            'field_name' => $field_name,
            'view_mode' => $build[$field_name]['#view_mode'],
          ]),
          '#attributes' => [
            'title' => $this->t('Reload content for this field'),
            'class' => [
              'use-ajax',
              'frontend-editing-update-content',
              'visually-hidden',
            ],
            'data-fe-update-content' => $selector,
          ],
        ];
      }
      $add_items = FALSE;
      $move_up_down = FALSE;
      if (!empty($build[$field_name]['#third_party_settings']) && !empty($build[$field_name]['#third_party_settings']['frontend_editing']['add_items'])) {
        $add_items = TRUE;
      }
      if (!empty($build[$field_name]['#third_party_settings']) && !empty($build[$field_name]['#third_party_settings']['frontend_editing']['move_up_down'])) {
        $move_up_down = TRUE;
      }
      if (($add_items || $move_up_down) && $entity->access('update') && !($entity instanceof RevisionableInterface && !$entity->isLatestRevision())) {
        foreach (Element::getVisibleChildren($build[$field_name]) as $delta) {
          $item = $build[$field_name][$delta];
          if (!empty($item['#pre_render'])) {
            if ($add_items) {
              $build[$field_name][$delta]['#parent_field_view_mode'] = $build[$field_name]['#view_mode'];
              $build[$field_name][$delta]['#pre_render'][] = [
                $this->fieldReferenceHelper,
                'addAddItemButtons',
              ];
            }
            if ($move_up_down) {
              $build[$field_name][$delta]['#parent_field_view_mode'] = $build[$field_name]['#view_mode'];
              $build[$field_name][$delta]['#pre_render'][] = [
                $this->fieldReferenceHelper,
                'addMoveButtons',
              ];
            }
            if ($ajax_content_update) {
              $build[$field_name][$delta]['#parent_field_view_mode'] = $build[$field_name]['#view_mode'];
              $build[$field_name][$delta]['#pre_render'][] = [
                $this->fieldReferenceHelper,
                'addAjaxContentUpdate',
              ];
            }
          }
        }
      }
    }

    if (empty(Element::getVisibleChildren($build['frontend_editing']))) {
      unset($build['frontend_editing']);
    }
  }

  /**
   * Check formatter third party settings.
   */
  private function checkThirdPartyFormatterSettings(array &$element) {
    if (!empty($element['#third_party_settings']['frontend_editing'])) {
      $supported_entity_types = $this->configFactory
        ->get('frontend_editing.settings')
        ->get('entity_types');
      // Check if the bundle, that field is attached to, is allowed for frontend
      // editing.
      if (empty($supported_entity_types[$element['#items']->getSetting('target_type')])) {
        unset($element['#third_party_settings']);
      }
    }
  }

}
