<?php

namespace Drupal\frontend_editing\Controller;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\frontend_editing\Ajax\ReloadWindowCommand;
use Drupal\frontend_editing\FieldReferenceHelperInterface;
use Drupal\frontend_editing\Form\EntityReferenceAddForm;
use Drupal\frontend_editing\FrontendEditingFormBuilderInterface;
use Drupal\user\UserDataInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Frontend editing form.
 *
 * @package Drupal\frontend_editing\Controller
 */
class FrontendEditingController extends ControllerBase {

  /**
   * Renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The user data service.
   *
   * @var \Drupal\user\UserDataInterface
   */
  protected $userData;

  /**
   * The field reference helper.
   *
   * @var \Drupal\frontend_editing\FieldReferenceHelperInterface
   */
  protected $fieldReferenceHelper;

  /**
   * The frontend editing form builder.
   *
   * @var \Drupal\frontend_editing\FrontendEditingFormBuilderInterface
   */
  protected $frontendEditingFormBuilder;

  /**
   * FrontendEditingController constructor.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   Renderer service.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data storage.
   * @param \Drupal\frontend_editing\FieldReferenceHelperInterface $field_reference_helper
   *   The field reference helper.
   * @param \Drupal\frontend_editing\FrontendEditingFormBuilderInterface $frontend_editing_form_builder
   *   The frontend editing form builder.
   */
  public function __construct(RendererInterface $renderer, EntityRepositoryInterface $entity_repository, UserDataInterface $userData, FieldReferenceHelperInterface $field_reference_helper, FrontendEditingFormBuilderInterface $frontend_editing_form_builder) {
    $this->renderer = $renderer;
    $this->entityRepository = $entity_repository;
    $this->userData = $userData;
    $this->fieldReferenceHelper = $field_reference_helper;
    $this->frontendEditingFormBuilder = $frontend_editing_form_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
      $container->get('entity.repository'),
      $container->get('user.data'),
      $container->get('frontend_editing.field_reference'),
      $container->get('frontend_editing.form_builder')
    );
  }

  /**
   * Toggle frontend editing functionality.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The ajax response.
   */
  public function toggle(Request $request) {
    if (!$request->isXmlHttpRequest()) {
      throw new NotFoundHttpException();
    }
    // Get current state.
    $current_state = (bool) $this->userData->get('frontend_editing', $this->currentUser()->id(), 'enabled');
    // Revert it, as requested.
    $new_state = !$current_state;
    // Set the new value.
    $this->userData->set('frontend_editing', $this->currentUser()->id(), 'enabled', $new_state);
    // Prepare the response.
    $response = new AjaxResponse();
    if ($new_state) {
      $message = $this->t('Frontend editing has been enabled.');
      $response->addCommand(new InvokeCommand('.frontend-editing-toggle-link', 'addClass', ['frontend-editing--enabled']));
      $response->addCommand(new InvokeCommand('.frontend-editing-toggle-link', 'text', [$this->t('On')]));
      $response->addCommand(new InvokeCommand('body', 'removeClass', ['frontend-editing--hidden']));
    }
    else {
      $message = $this->t('Frontend editing has been disabled.');
      $response->addCommand(new InvokeCommand('.frontend-editing-toggle-link', 'removeClass', ['frontend-editing--enabled']));
      $response->addCommand(new InvokeCommand('.frontend-editing-toggle-link', 'text', [$this->t('Off')]));
      $response->addCommand(new InvokeCommand('body', 'addClass', ['frontend-editing--hidden']));
    }
    $response->addCommand(new MessageCommand($message, NULL, ['type' => 'status']));
    $response->addCommand(new InvokeCommand('.frontend-editing-toggle-link', 'attr', [
      'data-toggle-state',
      $new_state,
    ]));
    $response->addCommand(new InvokeCommand('.frontend-editing-toggle-link', 'removeClass', ['frontend-editing-toggle-not-configured']));

    return $response;
  }

  /**
   * Implements form load request handler.
   *
   * @param string $type
   *   Entity type.
   * @param int $id
   *   Entity id.
   * @param string $operation
   *   Form operation.
   * @param string $display
   *   Form operation.
   *
   * @return array
   *   Form array.
   */
  public function getForm($type, $id, $operation = 'edit', $display = 'default') {
    // First check if given storage exists and can be loaded.
    try {
      $storage = $this->entityTypeManager()->getStorage($type);
    }
    catch (PluginNotFoundException | InvalidPluginDefinitionException $exception) {
      $this->messenger()->addError($exception->getMessage());
      return [];
    }
    // Load the entity.
    $entity = $storage->load($id);
    // Check if entity exists.
    if (!$entity instanceof ContentEntityInterface) {
      $this->messenger()->addWarning($this->t('Entity of type @type and id @id was not found',
        ['@type' => $type, '@id' => $id]
      ));
      return [];
    }
    // Remove all messages.
    $this->messenger()->deleteAll();
    // If the entity type is translatable, ensure we use the proper entity
    // translation for the current context, so that the access check is made on
    // the entity translation.
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $entity = $this->entityRepository->getTranslationFromContext($entity);
    $entityForm = $this->frontendEditingFormBuilder->buildForm($entity, $operation, $display);
    $entityForm['#attached']['library'][] = 'frontend_editing/forms_helper';
    return $entityForm;
  }

  /**
   * Checks access to add item to parent.
   *
   * @param string $bundle
   *   The bundle id.
   * @param string $parent_type
   *   The parent entity type.
   * @param mixed $parent
   *   The parent id.
   * @param string $parent_field_name
   *   The parent field name.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function accessAddType($bundle, $parent_type, $parent, $parent_field_name) {
    return $this->fieldReferenceHelper->accessAddType($bundle, $parent_type, $parent, $parent_field_name);
  }

  /**
   * Checks access to add paragraph to parent.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function accessAdd($parent_type, $parent, $parent_field_name) {
    return $this->fieldReferenceHelper->accessAdd($parent_type, $parent, $parent_field_name);
  }

  /**
   * Checks access to move item up request.
   *
   * @param string $parent_type
   *   The parent entity type.
   * @param mixed $parent
   *   The parent id.
   * @param string $parent_field_name
   *   The parent field name.
   * @param mixed $current_item
   *   The current item id.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function accessUp($parent_type, $parent, $parent_field_name, $current_item) {
    return $this->fieldReferenceHelper->accessMove($parent_type, $parent, $parent_field_name, $current_item, 'up');
  }

  /**
   * Checks access to move item down.
   *
   * @param string $parent_type
   *   The parent entity type.
   * @param mixed $parent
   *   The parent id.
   * @param string $parent_field_name
   *   The parent field name.
   * @param mixed $current_item
   *   The current item id.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function accessDown($parent_type, $parent, $parent_field_name, $current_item) {
    return $this->fieldReferenceHelper->accessMove($parent_type, $parent, $parent_field_name, $current_item, 'down');
  }

  /**
   * Checks access to update content.
   *
   * @param string $entity_type_id
   *   The entity type id.
   * @param int $entity_id
   *   The entity id.
   * @param string $field_name
   *   The field name.
   * @param string $view_mode
   *   The view mode.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function accessUpdateContent($entity_type_id, $entity_id, $field_name, $view_mode) {
    $entity = $this->entityTypeManager()->getStorage($entity_type_id)->load($entity_id);
    if (!$entity) {
      $result = AccessResult::forbidden('Entity does not exist.');
    }
    elseif (!$entity->access('view')) {
      $result = AccessResult::forbidden('You are not allowed to view this entity.');
    }
    elseif (!$entity->hasField($field_name)) {
      $result = AccessResult::forbidden('Entity has no field ' . $field_name . ' .');
    }
    elseif (!$entity->get($field_name)->access('view')) {
      $result = AccessResult::forbidden('You are not allowed to view field ' . $field_name . ' .');
    }
    else {
      $result = AccessResult::allowed();
    }
    return $result->addCacheableDependency($entity)->cachePerPermissions();
  }

  /**
   * Shift up a single paragraph.
   */
  public function up($parent_type, $parent, $parent_field_name, $current_item, Request $request) {
    $message = FALSE;
    if (!$this->fieldReferenceHelper->move($parent_type, $parent, $parent_field_name, $current_item, 'up')) {
      $message = $this->t('The paragraph could not be moved up.');
    }
    if ($request->isXmlHttpRequest()) {
      $view_mode = $request->query->get('view_mode_id', 'default');
      $response = $this->updateContent($parent_type, $parent, $parent_field_name, $view_mode, $request);
      if ($message) {
        $response->addCommand(new MessageCommand($message, NULL, ['type' => 'error']));
      }
      return $response;
    }
    if (!empty($message)) {
      $this->messenger()->addError($message);
    }
    return new RedirectResponse($request->query->get('destination'));
  }

  /**
   * Shift down a single paragraph.
   */
  public function down($parent_type, $parent, $parent_field_name, $current_item, Request $request) {
    $message = FALSE;
    if (!$this->fieldReferenceHelper->move($parent_type, $parent, $parent_field_name, $current_item, 'down')) {
      $message = $this->t('The paragraph could not be moved down.');
    }
    if ($request->isXmlHttpRequest()) {
      $view_mode = $request->query->get('view_mode_id', 'default');
      $response = $this->updateContent($parent_type, $parent, $parent_field_name, $view_mode, $request);
      if ($message) {
        $response->addCommand(new MessageCommand($message, NULL, ['type' => 'error']));
      }
      return $response;
    }
    if (!empty($message)) {
      $this->messenger()->addError($message);
    }
    return new RedirectResponse($request->query->get('destination'));
  }

  /**
   * Update content with ajax.
   *
   * @param string $entity_type_id
   *   The entity type id.
   * @param mixed $entity_id
   *   The entity id.
   * @param string $field_name
   *   The field name.
   * @param string $view_mode
   *   The view mode.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The ajax response.
   */
  public function updateContent($entity_type_id, $entity_id, $field_name, $view_mode, Request $request) {
    if (!$request->isXmlHttpRequest()) {
      throw new NotFoundHttpException();
    }
    $response = new AjaxResponse();
    if (empty($view_mode)) {
      $view_mode = 'default';
    }
    $entity = NULL;
    try {
      $entity = $this->entityTypeManager()
        ->getStorage($entity_type_id)
        ->load($entity_id);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $message = $this->t('Entity of type @type and id @id was not found',
        ['@type' => $entity_type_id, '@id' => $entity_id]
      );
      $response->addCommand(new MessageCommand($message, NULL, ['type' => 'error']));
    }
    if (!$entity) {
      $message = $this->t('Entity of type @type and id @id was not found',
        ['@type' => $entity_type_id, '@id' => $entity_id]
      );
      $response->addCommand(new MessageCommand($message, NULL, ['type' => 'error']));
    }
    // If there are errors, early return and reload the page.
    if (!empty($response->getCommands())) {
      $response->addCommand(new ReloadWindowCommand());
      return $response;
    }
    $entity = $this->entityRepository->getTranslationFromContext($entity);
    $updated_content = $entity->get($field_name)->view($view_mode);
    $this->addReferencePreRender($updated_content);
    $selector = '[data-frontend-editing="' . $entity_type_id . '--' . $entity_id . '--' . $field_name . '--' . $view_mode . '"]';
    $response->addCommand(new HtmlCommand($selector, $updated_content));
    return $response;
  }

  /**
   * Add reference pre render.
   *
   * @param array $updated_content
   *   The updated content.
   */
  protected function addReferencePreRender(array &$updated_content) {
    $add_items = FALSE;
    $move_up_down = FALSE;
    $ajax_content_update = FALSE;
    if (!empty($updated_content['#third_party_settings']) && !empty($updated_content['#third_party_settings']['frontend_editing']['add_items'])) {
      $add_items = TRUE;
    }
    if (!empty($updated_content['#third_party_settings']) && !empty($updated_content['#third_party_settings']['frontend_editing']['move_up_down'])) {
      $move_up_down = TRUE;
    }
    if (!empty($updated_content['#third_party_settings']) && !empty($updated_content['#third_party_settings']['frontend_editing']['ajax_content_update'])) {
      $ajax_content_update = TRUE;
    }
    if ($add_items || $move_up_down) {
      foreach (Element::getVisibleChildren($updated_content) as $delta) {
        $item = $updated_content[$delta];
        if (!empty($item['#pre_render'])) {
          if ($add_items) {
            $updated_content[$delta]['#parent_field_view_mode'] = $updated_content['#view_mode'];
            $updated_content[$delta]['#pre_render'][] = [
              $this->fieldReferenceHelper,
              'addAddItemButtons',
            ];
          }
          if ($move_up_down) {
            $updated_content[$delta]['#parent_field_view_mode'] = $updated_content['#view_mode'];
            $updated_content[$delta]['#pre_render'][] = [
              $this->fieldReferenceHelper,
              'addMoveButtons',
            ];
          }
          if ($ajax_content_update) {
            $updated_content[$delta]['#parent_field_view_mode'] = $updated_content['#view_mode'];
            $updated_content[$delta]['#pre_render'][] = [
              $this->fieldReferenceHelper,
              'addAjaxContentUpdate',
            ];
          }
        }
      }
    }
  }

  /**
   * Displays the list of bundles that are available for creation.
   *
   * The list is limited to the bundles that are allowed to be added to the
   * parent entity field.
   *
   * @param string $parent_type
   *   The parent entity type.
   * @param string $parent
   *   The parent id.
   * @param string $parent_field_name
   *   The parent field name.
   * @param int $current_item
   *   The current item id. The one that initiated the request.
   * @param int $before
   *   Could be 0 or 1. 1 means that the new paragraph should be added before
   *   current paragraph.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return array
   *   Render array with the list of bundles as links to add item
   *   form.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function itemAddPage($parent_type, $parent, $parent_field_name, $current_item, $before, Request $request) {
    try {
      $parent_entity = $this->entityTypeManager()->getStorage($parent_type)
        ->load($parent);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      throw new NotFoundHttpException('Parent entity not found.');
    }
    if (!$parent_entity || !$parent_entity->hasField($parent_field_name)) {
      throw new NotFoundHttpException('Parent entity not found.');
    }
    // By default, assume that all paragraphs are allowed.
    $allowed_bundles = NULL;
    // Check the field settings.
    $settings = $parent_entity->get($parent_field_name)->getSettings();
    if (!empty($settings['handler_settings']['target_bundles'])) {
      $target_bundles = $settings['handler_settings']['target_bundles'];
      $allowed_bundles = array_filter($target_bundles);
    }
    // Check if the allowed paragraphs should be negated.
    if (!empty($settings['handler_settings']['negate'])) {
      $allowed_bundles = array_filter($settings['handler_settings']['target_bundles_drag_drop'], function ($item) {
        return !$item['enabled'];
      });
      $allowed_bundles = array_keys($allowed_bundles);
    }
    $reference_entity_type = $this->entityTypeManager()->getDefinition($settings['target_type']);
    $allowed_bundles = $this->entityTypeManager()
      ->getStorage($reference_entity_type->get('bundle_entity_type'))
      ->loadMultiple($allowed_bundles);
    $items = [];
    foreach ($allowed_bundles as $allowed_bundle) {
      $create_bundle_access = $this->fieldReferenceHelper->accessAddType($allowed_bundle->id(), $parent_type, $parent, $parent_field_name);
      if (!$create_bundle_access->isAllowed()) {
        continue;
      }
      $item = [
        '#wrapper_attributes' => [
          'class' => [
            'frontend-editing-add-dialog-row',
          ],
        ],
      ];
      if ($allowed_bundle->getEntityTypeId() == 'paragraphs_type' && !empty($allowed_bundle->get('icon_default'))) {
        $item['icon'] = $this->buildIconElement($allowed_bundle);
        $item['#wrapper_attributes']['class'][] = 'with-icon';
      }

      $item += [
        '#type' => 'link',
        '#title' => $this->t('Add @type', ['@type' => $allowed_bundle->label()]),
        '#url' => Url::fromRoute('frontend_editing.entity_reference_item_add', [
          'parent_type' => $parent_type,
          'parent' => $parent,
          'parent_field_name' => $parent_field_name,
          'bundle' => $allowed_bundle->id(),
          'current_item' => $current_item,
          'before' => $before,
        ], [
          'query' => $request->query->all(),
        ]),
        '#attributes' => [
          'class' => [
            'js-form-submit',
            'form-submit',
            'button',
            'button--large',
            'field-add-more-submit',
          ],
          'name' => $parent_field_name . '_' . $allowed_bundle->id() . '_add_more',
        ],
      ];
      $items[$allowed_bundle->id()] = $item;
    }
    if (!empty($settings['handler_settings']['target_bundles_drag_drop'])) {
      uasort($settings['handler_settings']['target_bundles_drag_drop'], function ($a, $b) {
        return $a['weight'] <=> $b['weight'];
      });
      if (!empty($settings['handler_settings']['negate'])) {
        $settings['handler_settings']['target_bundles_drag_drop'] = array_filter($settings['handler_settings']['target_bundles_drag_drop'], function ($value) {
          return !$value['enabled'];
        });
      }
      else {
        $settings['handler_settings']['target_bundles_drag_drop'] = array_filter($settings['handler_settings']['target_bundles_drag_drop'], function ($value) {
          return $value['enabled'];
        });
      }
      $items = array_replace(array_flip(array_keys($settings['handler_settings']['target_bundles_drag_drop'])), $items);
      foreach ($items as $bundle => $item) {
        if (!is_array($item) || !isset($item['#type'])) {
          unset($items[$bundle]);
        }
      }
      $items = array_values($items);
    }

    // Attach filter bundles feature.
    $frontend_editing_settings = $this->config('frontend_editing.settings');
    if ($frontend_editing_settings->get('filter_add_items') && ($frontend_editing_settings->get('filter_add_items_threshold') <= count($items))) {
      array_unshift($items, [
        '#type' => 'textfield',
        '#attributes' => ['autofocus' => TRUE],
        '#name' => 'search',
        '#placeholder' => $this->t('Search'),
        '#wrapper_attributes' => [
          'class' => [
            'frontend-editing-add-dialog-search',
            'frontend-editing-add-dialog-row',
          ],
        ],
        '#attached' => [
          'library' => [
            'frontend_editing/filter_add_items',
          ],
        ],
      ]);
    }

    $build = [];
    $field_storage_definition = $parent_entity->get($parent_field_name)->getFieldDefinition()->getFieldStorageDefinition();
    if ($field_storage_definition->getType() == 'entity_reference') {
      $build['form_wrapper'] = [
        '#type' => 'details',
        '#title' => $this->t('Add existing @type', ['@type' => $field_storage_definition->getPropertyDefinition('entity')->getLabel()]),
      ];
      $build['form_wrapper']['form'] = $this->formBuilder()->getForm(EntityReferenceAddForm::class, $parent_entity->get($parent_field_name)->getFieldDefinition());
    }
    $build['add_form_links'] = [
      '#theme' => 'item_list',
      '#items' => $items,
      '#empty' => [
        '#markup' => $this->t('You are not allowed to create new items. Please check permissions.'),
      ],
      '#attributes' => [
        'class' => ['frontend-editing-add-dialog-list'],
      ],
      '#attached' => [
        'library' => [
          'frontend_editing/add_items',
          'frontend_editing/forms_helper',
        ],
      ],
    ];
    return $build;
  }

  /**
   * Add item form.
   *
   * @param string $bundle
   *   The bundle.
   * @param string $parent_type
   *   The parent entity type.
   * @param mixed $parent
   *   The parent id.
   * @param string $parent_field_name
   *   The parent field name.
   * @param int $current_item
   *   The current item id. The one that initiated the request.
   * @param int $before
   *   Could be 0 or 1. 1 means that the new paragraph should be added before
   *   current paragraph.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return array
   *   The form array.
   */
  public function itemAddForm($bundle, $parent_type, $parent, $parent_field_name, $current_item, $before, Request $request) {
    $parent_entity = $this->entityTypeManager()->getStorage($parent_type)->load($parent);
    $operation = 'add';
    $parent_field_definition = $parent_entity->get($parent_field_name)->getFieldDefinition();
    $settings = $parent_field_definition->getSettings();
    $entity_type_definition = $this->entityTypeManager()->getDefinition($settings['target_type']);
    $values = [
      $entity_type_definition->getKey('bundle') => $bundle,
    ];
    if ($settings['target_type'] == 'paragraph') {
      $operation = 'entity_add';
      $values['parent_field_name'] = $parent_field_name;
      $values['parent_type'] = $parent_type;
      $values['parent_id'] = $parent;
    }
    $entity = $this->entityTypeManager()->getStorage($entity_type_definition->id())->create($values);
    return $this->entityFormBuilder()->getForm($entity, $operation);
  }

  /**
   * Builds the icon element for a given Paragraphs type.
   *
   * This method generates a render array for the icon element using the value
   * from the 'icon_default' field. It assumes that an icon is present and
   * should only be called when an icon is defined.
   *
   * @param \Drupal\paragraphs\ParagraphsTypeInterface $paragraphs_type
   *   The Paragraphs type entity.
   *
   * @return array
   *   A render array for the icon element.
   */
  protected function buildIconElement(
    EntityInterface $paragraphs_type,
  ): array {
    $icon = $paragraphs_type->get('icon_default');
    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['paragraph-type-icon', $paragraphs_type->id()],
      ],
      'image' => [
        '#theme' => 'image',
        '#uri' => $icon,
        '#alt' => $paragraphs_type->label(),
      ],
    ];
  }

}
