<?php

namespace Drupal\frontend_editing_paragraphs;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Url;
use Drupal\frontend_editing\FrontendEditingFormBuilder;
use Drupal\frontend_editing\FrontendEditingFormBuilderInterface;
use Drupal\paragraphs\ParagraphInterface;
use Drupal\paragraphs_edit\ParagraphLineageInspector;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Decorates FrontendEditingFormBuilder class.
 */
class FrontendEditingParagraphsFormBuilder extends FrontendEditingFormBuilder {

  /**
   * The decorated service.
   *
   * @var \Drupal\frontend_editing\FrontendEditingFormBuilderInterface
   */
  protected $inner;

  /**
   * The paragraph lineage inspector.
   *
   * @var \Drupal\paragraphs_edit\ParagraphLineageInspector
   */
  protected $paragraphLineageInspector;

  /**
   * Constructs FrontendEditingFormBuilder.
   *
   * @param \Drupal\frontend_editing\FrontendEditingFormBuilderInterface $inner
   *   The decorated service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $entity_form_builder
   *   The entity form builder.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Drupal\paragraphs_edit\ParagraphLineageInspector $paragraph_lineage_inspector
   *   The paragraph lineage inspector.
   */
  public function __construct(FrontendEditingFormBuilderInterface $inner, EntityTypeManagerInterface $entity_type_manager, EntityFormBuilderInterface $entity_form_builder, FormBuilderInterface $form_builder, ParagraphLineageInspector $paragraph_lineage_inspector) {
    parent::__construct($entity_type_manager, $entity_form_builder, $form_builder);
    $this->inner = $inner;
    $this->paragraphLineageInspector = $paragraph_lineage_inspector;
  }

  /**
   * {@inheritDoc}
   */
  public function buildEditForm(ContentEntityInterface $entity, $operation = 'edit', $display = 'default', $form_state_additions = []) {
    if ($entity instanceof ParagraphInterface) {
      $operation = 'entity_edit';

      // Paragraphs cannot be saved through frontend editing when before the
      // save the user has interacted with the form in a way that it was
      // cached - e.g. by using AJAX to exchange an element or to add a new
      // element. An example is a block reference paragraph, where when
      // selecting a new reference from the select list an Ajax request will
      // be triggered.
      //
      // On submitting the form the cached form object will be used for
      // further processing. The problem is that the cached form object
      // (ParagraphEditForm) does not have the class property $root_parent
      // set as this is set only when accessing the form through the route
      // “paragraphs_edit.edit_form”, however the current implementation of
      // Frontend Editing only uses that route to submit the form to
      // (manipulates the form action before returning the form). AJAX
      // interactions with the form however go through the route
      // “xi_frontend_editing.form”, which misses the route parameter
      // “root_parent” and then the form object is cached without the
      // corresponding class property being set. The AJAX interactions are
      // routed through “xi_frontend_editing.form” because the paragraph
      // form is retrieved initially from that route and the AJAX system
      // uses the current route when building the ajax elements.
      // @see \Drupal\Core\Render\Element\RenderElement::preRenderAjaxForm()
      //
      // One solution is to ensure that the Frontend Editing passes the host
      // entity to the form build args when retrieving the form for the
      // paragraph. This however is still not a perfect solution, as the
      // “xi_frontend_editing.form” route will further be used for form
      // interactions, but the form will be routed somewhere else for
      // submission.
      $root_parent = $this->paragraphLineageInspector->getRootParent($entity);
      $form_state_additions = ['build_info' => ['args' => ['root_parent' => $root_parent]]];
    }
    $form = $this->inner->buildEditForm($entity, $operation, $display, $form_state_additions);
    if ($entity instanceof ParagraphInterface) {
      $url = Url::fromRoute('paragraphs_edit.edit_form', [
        'root_parent_type' => $root_parent->getEntityTypeId(),
        'root_parent' => $root_parent->id(),
        'paragraph' => $entity->id(),
      ]);
      $form['#action'] = $url->toString();
      $parent_field_name = $entity->get('parent_field_name')->value;
      $parent_entity = $entity->getParentEntity();
      $parent_field_definition = $parent_entity->get($parent_field_name)->getFieldDefinition();
      if ($parent_entity->isTranslatable() && !$parent_entity->isDefaultTranslation() && !$parent_field_definition->isTranslatable()) {
        $form['actions']['delete']['#access'] = FALSE;
      }
    }
    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function buildDeleteForm(ContentEntityInterface $entity, $operation = 'delete', $display = 'default', $form_state_additions = []) {
    if ($entity instanceof ParagraphInterface) {
      // By default, paragraph reference fields do not support translations.
      // For this reason we need to check in case the parent entity is
      // translatable that current translation is the default one and that
      // the paragraph field supports translations. In other case do not
      // allow to delete paragraph, because it will break the sync
      // translation and the paragraph will be deleted for all translations.
      // There are modules that allow async translation of paragraphs. In
      // this case it will still be possible to do, because then the field
      // definition will identify that the field is translatable.
      $parent_field_name = $entity->get('parent_field_name')->value;
      $parent_entity = $entity->getParentEntity();
      $parent_field_definition = $parent_entity->get($parent_field_name)->getFieldDefinition();
      if ($parent_entity->isTranslatable() && !$parent_entity->isDefaultTranslation() && !$parent_field_definition->isTranslatable()) {
        throw new AccessDeniedHttpException('You are not allowed to delete this paragraph, because paragraph parent field is not translatable.');
      }
      $operation = 'entity_delete';
      $root_parent = $this->paragraphLineageInspector->getRootParent($entity);
      $form_state_additions['build_info']['args']['root_parent'] = $root_parent;
    }
    $form = $this->inner->buildDeleteForm($entity, $operation, $display, $form_state_additions);
    if ($entity instanceof ParagraphInterface) {
      $url = Url::fromRoute('paragraphs_edit.delete_form', [
        'root_parent_type' => $root_parent->getEntityTypeId(),
        'root_parent' => $root_parent->id(),
        'paragraph' => $entity->id(),
      ]);
      $form['#action'] = $url->toString();
    }
    return $form;
  }

}
