<?php

namespace Drupal\frontend_editing\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure frontend_editing settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The extension list module.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $extensionListModule;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->fileUrlGenerator = $container->get('file_url_generator');
    $instance->extensionListModule = $container->get('extension.list.module');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'frontend_editing_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['frontend_editing.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('frontend_editing.settings');

    $form['width_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Width settings'),
    ];

    $form['width_settings']['sidebar_width'] = [
      '#title' => $this->t('Sidebar width'),
      '#type' => 'number',
      '#default_value' => $config->get('sidebar_width') ?? 30,
      '#description' => $this->t('Set the width of the editing sidebar when it opens. Minimum width is 30%.'),
      '#min' => 30,
      '#max' => 40,
      '#required' => TRUE,
    ];

    $form['width_settings']['full_width'] = [
      '#title' => $this->t('Full width'),
      '#type' => 'number',
      '#default_value' => $config->get('full_width') ?? 70,
      '#description' => $this->t('Set the width of the editing sidebar when it is expanded. Minimum width is 50%.'),
      '#min' => 50,
      '#max' => 95,
      '#required' => TRUE,
    ];

    // Preview of the hover highlight mode.
    $form['hover_highlight_preview'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['frontend-editing-hover-highlight-preview'],
      ],
    ];
    $form['hover_highlight_preview']['hover_highlight'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Hover Highlight Mode'),
      '#description' => $this->t('Toggle between a subtle border or a more distinct margin and background highlight for editable paragraphs on hover.'),
      '#default_value' => $config->get('hover_highlight'),
    ];
    $form['hover_highlight_preview']['disabled'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="hover_highlight"]' => ['checked' => FALSE],
        ],
      ],
    ];
    $form['hover_highlight_preview']['disabled']['image'] = [
      '#type' => 'html_tag',
      '#tag' => 'img',
      '#attributes' => [
        'src' => $this->fileUrlGenerator->generateAbsoluteString($this->extensionListModule->getPath('frontend_editing') . '/images/hover_highlight_disabled.png'),
        'alt' => $this->t('Hover Highlight Preview'),
        'width' => '400px',
        'height' => 'auto',
      ],
    ];
    $form['hover_highlight_preview']['enabled'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="hover_highlight"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['hover_highlight_preview']['enabled']['image'] = [
      '#type' => 'html_tag',
      '#tag' => 'img',
      '#attributes' => [
        'src' => $this->fileUrlGenerator->generateAbsoluteString($this->extensionListModule->getPath('frontend_editing') . '/images/hover_highlight_enabled.png'),
        'alt' => $this->t('Hover Highlight Preview'),
        'width' => '400px',
        'height' => 'auto',
      ],
    ];

    // Color picker field for the primary color.
    $form['primary_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Primary UI Color'),
      '#default_value' => $config->get('primary_color'),
      '#description' => $this->t('Choose the primary color for the frontend editing interface. Be sure to choose a color that has enough contrast with the background color of your site.'),
    ];

    // Enable on/off detach of the editor.
    $form['allow_detach'] = [
      '#type' => 'checkbox',
      '#default_value' => $config->get('allow_detach'),
      '#title' => $this->t('Allow detach'),
      '#description' => $this->t('Allows detaching of sidebar to a child browser window.'),
    ];

    // Enable on/off toggle in UI.
    $form['ui_toggle'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable on/off toggle in UI'),
      '#description' => $this->t('When enabled, the user will be able to toggle the frontend editing on and off via a button in the UI.'),
      '#default_value' => $config->get('ui_toggle'),
    ];

    // Wrapper for UI Toggle settings.
    $form['ui_toggle_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('UI Toggle Settings'),
      '#open' => $config->get('ui_toggle'),
      '#description' => $this->t('Configure the UI Toggle settings.'),
      '#tree' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="ui_toggle"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $offsets = [
      'top' => 'Top',
      'right' => 'Right',
      'bottom' => 'Bottom',
      'left' => 'Left',
    ];

    foreach ($offsets as $key => $label) {
      $form['ui_toggle_settings']["offset_$key"] = [
        '#type' => 'number',
        '#title' => $this->t("Offset @label", ['@label' => $label]),
        '#description' => $this->t("Define the offset @label from the selected position. (px)", ['@label' => $label]),
        '#default_value' => $config->get("ui_toggle_settings.offset_$key"),
      ];
    }

    $form['action_links_in_viewport'] = [
      '#type' => 'radios',
      '#title' => $this->t('Keep action links in viewport:'),
      '#options' => [
        '_none' => $this->t('No'),
        'duplicate' => $this->t('Duplicate action links at the bottom of the element'),
        'floating' => $this->t('Make action links sticky'),
      ],
      '#description' => $this->t('If element to edit is too long, this helps to keep the actions link visible (experimental, use with caution).'),
      '#default_value' => $config->get('action_links_in_viewport'),
    ];

    // Wrapper additional float action links button at bottom settings.
    $form['duplicate_action_links_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Additional Action Links Settings'),
      '#open' => $config->get('action_links_in_viewport') == 'duplicate',
      '#description' => $this->t('Configure the additional action links settings.'),
      '#tree' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="action_links_in_viewport"]' => ['value' => 'duplicate'],
        ],
      ],
    ];

    $form['duplicate_action_links_settings']['height'] = [
      '#title' => $this->t('Height'),
      '#type' => 'number',
      '#default_value' => $config->get('duplicate_action_links.height') ?? 300,
      '#description' => $this->t('Set the container height for which additional action links at bottom will be displayed. Minimum height is 300px.'),
      '#min' => 300,
      '#states' => [
        'required' => [
          ':input[name="action_links_in_viewport"]' => ['value' => 'duplicate'],
        ],
      ],
    ];

    // Filter add items.
    $form['filter_add_items'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['frontend-editing-filter-add-items'],
      ],
    ];
    $form['filter_add_items']['filter_add_items'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable filtering of new items.'),
      '#description' => $this->t('Adds a search input to filter entity reference types.'),
      '#default_value' => $config->get('filter_add_items'),
    ];

    $form['filter_add_items']['filter_add_items_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Filter add items threshold.'),
      '#description' => $this->t('Enable add new items filtering when their number exceeds this value.'),
      '#default_value' => $config->get('filter_add_items_threshold') ?? 10,
      '#min' => 0,
      '#states' => [
        'visible' => [
          ':input[name="filter_add_items"]' => ['checked' => TRUE],
        ],
      ],
    ];

    if ($this->moduleHandler->moduleExists('preview')) {
      $form['preview_settings'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Preview settings'),
      ];

      $form['preview_settings']['automatic_preview'] = [
        '#title' => $this->t('Automatic preview'),
        '#type' => 'checkbox',
        '#default_value' => $config->get('automatic_preview'),
        '#description' => $this->t('Enable live preview when editing entities.'),
      ];
      $form['preview_settings']['automatic_preview_details'] = [
        '#markup' => $this->t('Make sure that entity types and bundles that you want to preview are enabled <a href="@url" target="_blank">here</a>.', ['@url' => Url::fromRoute('preview.settings')->toString()]),
      ];
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('frontend_editing.settings')
      ->set('sidebar_width', $form_state->getValue('sidebar_width'))
      ->set('full_width', $form_state->getValue('full_width'))
      ->set('automatic_preview', (bool) $form_state->getValue('automatic_preview'))
      ->set('hover_highlight', (bool) $form_state->getValue('hover_highlight'))
      ->set('primary_color', $form_state->getValue('primary_color'))
      ->set('allow_detach', (bool) $form_state->getValue('allow_detach'))
      ->set('ui_toggle', (bool) $form_state->getValue('ui_toggle'))
      ->set('ui_toggle_settings', $form_state->getValue('ui_toggle_settings'))
      ->set('action_links_in_viewport', $form_state->getValue('action_links_in_viewport'))
      ->set('duplicate_action_links', [
        'height' => $form_state->getValue(['duplicate_action_links_settings', 'height']),
      ])
      ->set('filter_add_items', (bool) $form_state->getValue('filter_add_items'))
      ->set('filter_add_items_threshold', $form_state->getValue('filter_add_items_threshold'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
