<?php

declare(strict_types=1);

namespace Drupal\frontend_editing;

use Drupal\Component\Utility\Html;
use Drupal\Core\Render\ElementInfoManagerInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\user\UserDataInterface;

/**
 * Defines a class for lazy building render arrays.
 *
 * @internal
 */
final class ToolbarItem implements TrustedCallbackInterface {

  use StringTranslationTrait;

  /**
   * Constructs LazyBuilders object.
   *
   * @param \Drupal\Core\Render\ElementInfoManagerInterface $elementInfo
   *   Element info.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   */
  public function __construct(
    protected ElementInfoManagerInterface $elementInfo,
    protected UserDataInterface $userData,
    protected AccountProxyInterface $currentUser,
  ) {}

  /**
   * Render announcements.
   *
   * @return array
   *   Render array.
   */
  public function renderToggle(): array {
    $toggle_state = $this->userData->get(
      'frontend_editing',
      $this->currentUser->id(),
      'enabled'
    );
    $active_class = $toggle_state ? 'frontend-editing--enabled' : '';

    $build = [
      '#type' => 'link',
      '#url' => Url::fromRoute('frontend_editing.toggle'),
      '#cache' => [
        'context' => ['user.permissions'],
      ],
      '#title' => $toggle_state ? $this->t('On') : $this->t('Off'),
      '#id' => Html::getId('frontend-editing-toolbar-toggle-link'),
      '#attributes' => [
        'title' => $this->t('Enable frontend editing actions'),
        'class' => [
          'use-ajax',
          'frontend-editing-toggle-link',
          $active_class,
        ],
      ],
      '#attached' => [
        'library' => [
          'core/drupal.ajax',
          'frontend_editing/ui_toggle',
        ],
      ],
    ];

    // The renderer has already added element defaults by the time the lazy
    // builder is run.
    // @see https://www.drupal.org/project/drupal/issues/2609250
    $build += $this->elementInfo->getInfo('link');
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks(): array {
    return ['renderToggle', 'removeTabAttributes'];
  }

  /**
   * Render callback.
   */
  public static function removeTabAttributes(array $element): array {
    unset($element['tab']['#attributes']);
    return $element;
  }

}
