<?php

namespace Drupal\Tests\frontify_assets\Unit\Form;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\frontify_assets\Form\SettingsForm;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Config\TypedConfigManagerInterface;

/**
 * Tests the SettingsForm class.
 *
 * @group frontify_assets
 *
 * @coversDefaultClass \Drupal\frontify_assets\Form\SettingsForm
 */
class SettingsFormTest extends UnitTestCase {

  /**
   * The form being tested.
   *
   * @var \Drupal\frontify_assets\Form\SettingsForm
   */
  protected $form;

  /**
   * Mock configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * Mock config object.
   *
   * @var \Drupal\Core\Config\Config|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $config;

  /**
   * Mock messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $messenger;

  /**
   * Mock translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $stringTranslation;

  /**
   * Mock config Manager service.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $typedConfigManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->config = $this->createMock(Config::class);
    $this->messenger = $this->createMock(MessengerInterface::class);
    $this->stringTranslation = $this->createMock(TranslationInterface::class);
    $this->typedConfigManager = $this->createMock(TypedConfigManagerInterface::class);
    $this->configFactory->method('getEditable')
      ->with('frontify_assets.settings')
      ->willReturn($this->config);

    $this->stringTranslation->method('translate')
      ->willReturnCallback(function ($string) {
        return $string;
      });

    $this->form = new SettingsForm($this->configFactory,$this->typedConfigManager);
    $this->form->setStringTranslation($this->stringTranslation);
    $this->form->setMessenger($this->messenger);
  }

  /**
   * Test getFormId method.
   *
   * @covers ::getFormId
   */
  public function testGetFormId(): void {
    $this->assertEquals('frontify_assets_settings', $this->form->getFormId());
  }

  /**
   * Test getEditableConfigNames method.
   *
   * @covers ::getEditableConfigNames
   */
  public function testGetEditableConfigNames(): void {
    $reflection = new \ReflectionClass($this->form);
    $method = $reflection->getMethod('getEditableConfigNames');
    $method->setAccessible(TRUE);
    $expected = ['frontify_assets.settings'];
    $this->assertEquals($expected, $method->invoke($this->form));
  }

  /**
   * Test buildForm method.
   *
   * @covers ::buildForm
   */
  public function testBuildForm(): void {
    $form_state = $this->createMock(FormStateInterface::class);

    $this->config->method('get')
      ->willReturnMap([
        ['frontify_api_url', 'https://test.frontify.com'],
        ['frontify_client_id', 'client-test123'],
        ['dynamic_urls', TRUE],
        ['debug_mode', FALSE],
      ]);

    $form = $this->form->buildForm([], $form_state);

    // Top-level sections.
    $this->assertArrayHasKey('frontify_api', $form);
    $this->assertArrayHasKey('dynamic_urls', $form);
    $this->assertArrayHasKey('debug_settings', $form);

    // Test frontify_api section.
    $this->assertEquals('details', $form['frontify_api']['#type']);
    $this->assertTrue($form['frontify_api']['#open']);
    $this->assertArrayHasKey('url', $form['frontify_api']);
    $this->assertArrayHasKey('client_id', $form['frontify_api']);

    // Test URL field.
    $this->assertEquals('url', $form['frontify_api']['url']['#type']);
    $this->assertTrue($form['frontify_api']['url']['#required']);
    $this->assertEquals('https://test.frontify.com', $form['frontify_api']['url']['#default_value']);

    // Test client ID field.
    $this->assertEquals('textfield', $form['frontify_api']['client_id']['#type']);
    $this->assertTrue($form['frontify_api']['client_id']['#required']);
    $this->assertEquals('client-test123', $form['frontify_api']['client_id']['#default_value']);

    // Test dynamic_urls checkbox.
    $this->assertEquals('checkbox', $form['dynamic_urls']['#type']);
    $this->assertTrue($form['dynamic_urls']['#default_value']);

    // Test debug settings.
    $this->assertEquals('details', $form['debug_settings']['#type']);
    $this->assertArrayHasKey('debug_mode', $form['debug_settings']);
    $this->assertEquals('checkbox', $form['debug_settings']['debug_mode']['#type']);
    $this->assertFalse($form['debug_settings']['debug_mode']['#default_value']);

    // Test actions placement.
    $this->assertArrayHasKey('creds', $form);
    $this->assertArrayHasKey('actions', $form['creds']);
    $this->assertArrayNotHasKey('actions', $form);
  }

  /**
   * Test submitForm method.
   *
   * @covers ::submitForm
   */
  public function testSubmitForm(): void {
    $form_state = $this->createMock(FormStateInterface::class);

    $form_state->method('getValue')
    ->willReturnMap([
      ['url', 'https://new.frontify.com'],
      ['client_id', 'client-new456'],
      ['dynamic_urls', FALSE],
      ['debug_mode', TRUE],
    ]);

    $expected = [
      ['frontify_api_url', 'https://new.frontify.com'],
      ['frontify_client_id', 'client-new456'],
      ['dynamic_url', FALSE],
      ['debug_mode', TRUE],
    ];

    $this->config->expects($this->exactly(count($expected)))
      ->method('set')
      ->willReturnCallback(function ($key, $value) use (&$expected) {
      // Shift the next expected pair off the list
      $next = array_shift($expected);
      return $this; // because Config::set() is chainable
    });

    $this->config->expects($this->once())
      ->method('save')
      ->willReturnSelf();

    $form = [];
    $this->form->submitForm($form, $form_state);
  }

}