<?php

namespace Drupal\Tests\frontify_assets\Unit\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinitionInterface;
use Drupal\Core\Url;
use Drupal\frontify_assets\Plugin\Field\FieldType\FrontifyVideoField;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the FrontifyVideoField field type.
 *
 * @group frontify_assets
 *
 * @coversDefaultClass \Drupal\frontify_assets\Plugin\Field\FieldType\FrontifyVideoField
 */
class FrontifyVideoFieldTest extends UnitTestCase {

  /**
   * The field type being tested.
   *
   * @var \Drupal\frontify_assets\Plugin\Field\FieldType\FrontifyVideoField
   */
  protected $fieldType;

  /**
   * Mock field storage definition.
   *
   * @var \Drupal\Core\Field\FieldStorageDefinitionInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $fieldStorageDefinition;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->fieldStorageDefinition = $this->createMock(FieldStorageDefinitionInterface::class);
    
    $this->fieldType = $this->getMockBuilder(FrontifyVideoField::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['get', 'getSetting'])
      ->getMock();
  }

  /**
   * Test propertyDefinitions method.
   *
   * @covers ::propertyDefinitions
   */
  public function testPropertyDefinitions(): void {
    $properties = FrontifyVideoField::propertyDefinitions($this->fieldStorageDefinition);

    $this->assertArrayHasKey('uri', $properties);
    $this->assertArrayHasKey('download_uri', $properties);
    $this->assertArrayHasKey('alt', $properties);
    $this->assertArrayHasKey('filesize', $properties);
    $this->assertArrayHasKey('mimetype', $properties);
    $this->assertArrayHasKey('filename', $properties);
    $this->assertArrayHasKey('created_at', $properties);

    $this->assertInstanceOf(DataDefinitionInterface::class, $properties['uri']);
    $this->assertInstanceOf(DataDefinitionInterface::class, $properties['download_uri']);
  }

  /**
   * Test schema method.
   *
   * @covers ::schema
   */
  public function testSchema(): void {
    $schema = FrontifyVideoField::schema($this->fieldStorageDefinition);

    $this->assertArrayHasKey('columns', $schema);
    $this->assertArrayHasKey('indexes', $schema);

    $columns = $schema['columns'];
    $this->assertArrayHasKey('uri', $columns);
    $this->assertArrayHasKey('download_uri', $columns);
    $this->assertArrayHasKey('alt', $columns);
    $this->assertArrayHasKey('filesize', $columns);
    $this->assertArrayHasKey('mimetype', $columns);
    $this->assertArrayHasKey('filename', $columns);
    $this->assertArrayHasKey('created_at', $columns);

    $this->assertEquals('varchar', $columns['uri']['type']);
    $this->assertEquals(2048, $columns['uri']['length']);
    $this->assertEquals(255, $columns['alt']['length']);
  }

  /**
   * Test defaultFieldSettings method.
   *
   * @covers ::defaultFieldSettings
   */
  public function testDefaultFieldSettings(): void {
    $settings = FrontifyVideoField::defaultFieldSettings();

    $this->assertArrayHasKey('extension', $settings);
    $this->assertEquals('mov,mp4', $settings['extension']);
    $this->assertArrayNotHasKey('description_field', $settings);
  }

  /**
   * Test fieldSettingsForm method.
   *
   * @covers ::fieldSettingsForm
   */
  public function testFieldSettingsForm(): void {
    $form = [];
    $form_state = $this->createMock(FormStateInterface::class);

    $this->fieldType->method('getSetting')
      ->with('extension')
      ->willReturn('mov,mp4,avi');

    $element = $this->fieldType->fieldSettingsForm($form, $form_state);

    $this->assertArrayHasKey('extension', $element);
    $this->assertEquals('textfield', $element['extension']['#type']);
    $this->assertEquals('mov,mp4,avi', $element['extension']['#default_value']);
  }

  /**
   * Test validateExtension method with valid input.
   *
   * @covers ::validateExtension
   */
  public function testValidateExtensionValid(): void {
    $element = ['#value' => 'mov,mp4,avi', '#title' => 'Extension'];
    $form_state = $this->createMock(FormStateInterface::class);

    $form_state->expects($this->never())
      ->method('setError');

    FrontifyVideoField::validateExtension($element, $form_state);
  }

  /**
   * Test validateExtension method with invalid input.
   *
   * @covers ::validateExtension
   */
  public function testValidateExtensionInvalid(): void {
    $element = ['#value' => 'mov,mp4@avi', '#title' => 'Extension'];
    $form_state = $this->createMock(FormStateInterface::class);

    $form_state->expects($this->once())
      ->method('setError')
      ->with($element, $this->isInstanceOf(TranslatableMarkup::class));

    FrontifyVideoField::validateExtension($element, $form_state);
  }

  /**
   * Test isEmpty method with empty value.
   *
   * @covers ::isEmpty
   */
  public function testIsEmptyWithEmptyValue(): void {
    $uriProperty = $this->createMock(\Drupal\Core\TypedData\TypedDataInterface::class);
    $uriProperty->method('getValue')->willReturn('');

    $this->fieldType->method('get')
      ->with('uri')
      ->willReturn($uriProperty);

    $this->assertTrue($this->fieldType->isEmpty());
  }

  /**
   * Test isEmpty method with null value.
   *
   * @covers ::isEmpty
   */
  public function testIsEmptyWithNullValue(): void {
    $uriProperty = $this->createMock(\Drupal\Core\TypedData\TypedDataInterface::class);
    $uriProperty->method('getValue')->willReturn(NULL);

    $this->fieldType->method('get')
      ->with('uri')
      ->willReturn($uriProperty);

    $this->assertTrue($this->fieldType->isEmpty());
  }

  /**
   * Test isEmpty method with non-empty value.
   *
   * @covers ::isEmpty
   */
  public function testIsEmptyWithValue(): void {
    $uriProperty = $this->createMock(\Drupal\Core\TypedData\TypedDataInterface::class);
    $uriProperty->method('getValue')->willReturn('https://example.com/video.mp4');

    $this->fieldType->method('get')
      ->with('uri')
      ->willReturn($uriProperty);

    $this->assertFalse($this->fieldType->isEmpty());
  }

  /**
   * Test mainPropertyName method.
   *
   * @covers ::mainPropertyName
   */
  public function testMainPropertyName(): void {
    $this->assertEquals('uri', FrontifyVideoField::mainPropertyName());
  }

  /**
   * Test isExternal method.
   *
   * @covers ::isExternal
   */
  public function testIsExternal(): void {
    $url = $this->createMock(Url::class);
    $url->method('isExternal')->willReturn(TRUE);

    $fieldType = $this->getMockBuilder(FrontifyVideoField::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getUrl'])
      ->getMock();

    $fieldType->method('getUrl')->willReturn($url);

    $this->assertTrue($fieldType->isExternal());
  }

  /**
   * Test getUrl method.
   *
   * @covers ::getUrl
   */
  public function testGetUrl(): void {
    $fieldType = $this->getMockBuilder(FrontifyVideoField::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getUrl'])
      ->getMock();

    $url = $this->createMock(Url::class);
    $fieldType->method('getUrl')->willReturn($url);

    $result = $fieldType->getUrl();
    $this->assertInstanceOf(Url::class, $result);
  }

}