<?php

namespace Drupal\Tests\frontify_assets\Unit\Plugin\Validation\Constraint;

use Drupal\frontify_assets\Plugin\Validation\Constraint\FrontifyImageUrlConstraintValidator;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface;
use Symfony\Component\Validator\Constraint;

/**
 * @coversDefaultClass \Drupal\frontify_assets\Plugin\Validation\Constraint\FrontifyImageUrlConstraintValidator
 *
 * @group frontify_assets
 */
class FrontifyImageUrlConstraintValidatorTest extends TestCase {

  /**
   * Tests that a valid external URL passes without violations.
   */
  public function testValidExternalUrl(): void {
    $urlMock = $this->createMock(\Drupal\Core\Url::class);
    $urlMock->expects($this->once())
      ->method('isExternal')
      ->willReturn(TRUE);

    $linkItemMock = $this->getMockBuilder(\stdClass::class)
      ->addMethods(['getUrl'])
      ->getMock();
    $linkItemMock->expects($this->once())
      ->method('getUrl')
      ->willReturn($urlMock);

    $constraint = new class() extends Constraint {
      public $message = 'Invalid URL: @uri';
    };

    $context = $this->createMock(ExecutionContextInterface::class);
    $context->expects($this->never())
      ->method('addViolation');

    $validator = new FrontifyImageUrlConstraintValidator();
    $validator->initialize($context);
    $validator->validate($linkItemMock, $constraint);
  }

  /**
   * Tests that an invalid URL triggers a violation.
   */
  public function testInvalidUrl(): void {
    $urlMock = $this->createMock(\Drupal\Core\Url::class);
    $urlMock->expects($this->once())
      ->method('isExternal')
      ->willReturn(FALSE);

    $linkItemMock = $this->getMockBuilder(\stdClass::class)
      ->addMethods(['getUrl'])
      ->getMock();
    $linkItemMock->expects($this->once())
      ->method('getUrl')
      ->willReturn($urlMock);
    $linkItemMock->uri = 'internal:/path';

    $constraint = new class() extends \Symfony\Component\Validator\Constraint {
      public $message = 'Invalid URL: @uri';
    };

    $context = $this->createMock(\Symfony\Component\Validator\Context\ExecutionContextInterface::class);
    $context->expects($this->once())
      ->method('addViolation')
      ->with($constraint->message, ['@uri' => $linkItemMock->uri]);

    $validator = new \Drupal\frontify_assets\Plugin\Validation\Constraint\FrontifyImageUrlConstraintValidator();
    $validator->initialize($context);
    $validator->validate($linkItemMock, $constraint);
  }

  /**
   * Tests that an exception in getUrl() triggers a violation.
   */
  public function testGetUrlThrowsException(): void {
    $linkItemMock = $this->getMockBuilder(\stdClass::class)
      ->addMethods(['getUrl'])
      ->getMock();
    $linkItemMock->expects($this->once())
      ->method('getUrl')
      ->willThrowException(new \InvalidArgumentException());
    $linkItemMock->uri = 'broken:/uri';

    $constraint = new class() extends \Symfony\Component\Validator\Constraint {
      public $message = 'Invalid URL: @uri';
    };

    $context = $this->createMock(\Symfony\Component\Validator\Context\ExecutionContextInterface::class);
    $context->expects($this->once())
      ->method('addViolation')
      ->with($constraint->message, ['@uri' => $linkItemMock->uri]);

    $validator = new \Drupal\frontify_assets\Plugin\Validation\Constraint\FrontifyImageUrlConstraintValidator();
    $validator->initialize($context);
    $validator->validate($linkItemMock, $constraint);
  }

}
