<?php

namespace Drupal\Tests\frontify_assets\Unit\Service;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\frontify_assets\Service\FrontifyService;
use Drupal\image\ImageStyleInterface;
use Drupal\Tests\UnitTestCase;

/**
 * Test the frontify service.
 *
 * @group frontify_assets
 * @group Drupal
 *
 * @coverDefaultClass \Drupal\frontify_assets\Service\FrontifyService
 */
class FrontifyServiceTest extends UnitTestCase {

  /**
   * The frontify service.
   *
   * @var \Drupal\frontify_assets\Service\FrontifyService|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $service;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The image style storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $imageStyleStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->imageStyleStorage = $this->createMock(EntityStorageInterface::class);

    $this->entityTypeManager->method('getStorage')
      ->with('image_style')
      ->willReturn($this->imageStyleStorage);

    $this->service = new FrontifyService($this->entityTypeManager);
  }

  /**
   * Test the constructor.
   *
   * @covers ::__construct
   */
  public function testConstruct(): void {
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $imageStyleStorage = $this->createMock(EntityStorageInterface::class);

    $entityTypeManager->expects($this->once())
      ->method('getStorage')
      ->with('image_style')
      ->willReturn($imageStyleStorage);

    $service = new FrontifyService($entityTypeManager);

    $reflection = new \ReflectionClass($service);

    $entityTypeManagerProperty = $reflection->getProperty('entityTypeManager');
    $entityTypeManagerProperty->setAccessible(TRUE);
    $this->assertSame($entityTypeManager, $entityTypeManagerProperty->getValue($service));

    $imageStyleStorageProperty = $reflection->getProperty('imageStyleStorage');
    $imageStyleStorageProperty->setAccessible(TRUE);
    $this->assertSame($imageStyleStorage, $imageStyleStorageProperty->getValue($service));
  }

  /**
   * Tests the attachImageWidth with width only.
   *
   * @covers ::attachImageWidth
   * @covers ::extractUrlQueryString
   */
  public function testAttachImageWidthWithWidthOnly(): void {
    $uri = 'https://example.com/image.jpg';
    $result = $this->service->attachImageWidth($uri, 300);

    $expected = 'https://example.com/image.jpg?width=300&crop=fp&fp=0.5%2C0.5&fp_zoom=1';
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests the attachImageWidth with width and height.
   *
   * @covers ::attachImageWidth
   * @covers ::extractUrlQueryString
   */
  public function testAttachImageWidthWithWidthAndHeight(): void {
    $uri = 'https://example.com/image.jpg';
    $result = $this->service->attachImageWidth($uri, 300, 200);

    $expected = 'https://example.com/image.jpg?width=300&height=200&crop=fp&fp=0.5%2C0.5&fp_zoom=1';
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests the attachImageWidth with preview mode.
   *
   * @covers ::attachImageWidth
   * @covers ::extractUrlQueryString
   */
  public function testAttachImageWidthWithPreview(): void {
    $uri = 'https://example.com/image.jpg';
    $result = $this->service->attachImageWidth($uri, 300, 200, TRUE);

    $expected = 'https://example.com/image.jpg?width=300&height=200';
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests the attachImageWidth with existing query parameters.
   *
   * @covers ::attachImageWidth
   * @covers ::extractUrlQueryString
   */
  public function testAttachImageWidthWithExistingQuery(): void {
    $uri = 'https://example.com/image.jpg?existing=param';
    $result = $this->service->attachImageWidth($uri, 300);

    $expected = 'https://example.com/image.jpg?existing=param&width=300&crop=fp&fp=0.5%2C0.5&fp_zoom=1';
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests the attachImageWidth with no dimensions.
   *
   * @covers ::attachImageWidth
   * @covers ::extractUrlQueryString
   */
  public function testAttachImageWidthWithNoDimensions(): void {
    $uri = 'https://example.com/image.jpg';
    $result = $this->service->attachImageWidth($uri);

    $this->assertEquals($uri, $result);
  }


  /**
   * Tests the attachImageStyle with invalid style.
   *
   * @covers ::attachImageStyle
   * @covers ::extractUrlQueryString
   */
  public function testAttachImageStyleWithValidStyle(): void {
    $uri = 'https://example.com/image.jpg';

    $imageStyle = $this->createMockImageStyle(400, 300);

    $this->imageStyleStorage->method('load')
      ->with('test_style')
      ->willReturn($imageStyle);

    $result = $this->service->attachImageStyle($uri, 'test_style');

    $expected = 'https://example.com/image.jpg?width=400&height=300&crop=fp&fp=0.5%2C0.5&fp_zoom=1';
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests the attachImageStyle with non-existing image style.
   *
   * @covers ::attachImageStyle
   * @covers ::extractUrlQueryString
   */
  public function testAttachImageStyleWithNonExistingStyle(): void {
    $uri = 'https://example.com/image.jpg';

    $this->imageStyleStorage->method('load')
      ->with('non_existent')
      ->willReturn(NULL);

    $result = $this->service->attachImageStyle($uri, 'non_existent');

    $this->assertEquals($uri, $result);
  }

  /**
   * Tests the attachImageStyle with no image style.
   *
   * @covers ::attachImageStyle
   */
  public function testAttachImageStyleWithNoStyle(): void {
    $uri = 'https://example.com/image.jpg';

    $result = $this->service->attachImageStyle($uri);

    $this->assertEquals($uri, $result);
  }

  /**
   * Tests the attachImageStyle with existing query parameters.
   *
   * @covers ::attachImageStyle
   * @covers ::extractUrlQueryString
   */
  public function testAttachImageStyleWithExistingQuery(): void {
    $uri = 'https://example.com/image.jpg?existing=param';
    $imageStyle = $this->createMockImageStyle(400, 300);

    $this->imageStyleStorage->method('load')
      ->with('test_style')
      ->willReturn($imageStyle);

    $result = $this->service->attachImageStyle($uri, 'test_style');

    $expected = 'https://example.com/image.jpg?existing=param&width=400&height=300&crop=fp&fp=0.5%2C0.5&fp_zoom=1';
    $this->assertEquals($expected, $result);
  }

  /**
   * Creates a mock image style.
   *
   * @param int $width
   *   The width.
   * @param int $height
   *   The height.
   *
   * @return \PHPUnit\Framework\MockObject\MockObject|ImageStyleInterface
   *   The mock image style.
   */
  private function createMockImageStyle(int $width, int $height) {
    $imageStyle = $this->createMock(ImageStyleInterface::class);
    $effects = $this->getMockBuilder('stdClass')
      ->addMethods(['getConfiguration'])
      ->getMock();

    $effectConfig = [
      'test_effect' => [
        'data' => [
          'width' => $width,
          'height' => $height,
        ],
      ],
    ];

    $effects->method('getConfiguration')->willReturn($effectConfig);

    $imageStyle->method('getEffects')->willReturn($effects);

    return $imageStyle;
  }

}