<?php

namespace Drupal\full_screen_mode\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides the Full Screen Mode.
 *
 * @Block(
 *   id="full_screen_mode",
 *   admin_label = @Translation("Full Screen Mode"),
 * )
 */
class FullScreenModeBlock extends BlockBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'full_screen_mode' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    $form['button'] = [
      '#type' => 'details',
      '#title' => $this->t('Button configuration'),
      '#open' => TRUE,
      '#description' => $this->t('Configure the button property'),
    ];

    $options = [
      'top_right' => $this->t('Top right'),
      'top_left' => $this->t('Top left'),
      'bottom_right' => $this->t('Bottom right'),
      'bottom_left' => $this->t('Bottom left'),
    ];

    $form['button']['position'] = [
      '#type' => 'select',
      '#required' => TRUE,
      '#options' => $options,
      '#title' => $this->t('Button position'),
      '#description' => $this->t('This property controls how the button is positioned within the page.'),
      '#default_value' => $config['position'] ?? 'top_right',
    ];

    $form['button']['top'] = [
      '#type' => 'number',
      '#title' => $this->t('Top'),
      '#description' => $this->t('This property controls how the button is positioned relative to top of the page.'),
      '#default_value' => $config['top'] ?? '10',
      '#states' => [
        'visible' => [
          ':input[name="settings[button][position]"]' => [
            ['value' => 'top_right'],
            ['value' => 'top_left'],
          ],
        ],
      ],
      '#min' => 0,
      '#max' => 1025,
    ];

    $form['button']['bottom'] = [
      '#type' => 'number',
      '#title' => $this->t('Bottom'),
      '#description' => $this->t('This property controls how the button is positioned relative to bottom of the page.'),
      '#default_value' => $config['bottom'] ?? '0',
      '#states' => [
        'visible' => [
          ':input[name="settings[button][position]"]' => [
            ['value' => 'bottom_right'],
            ['value' => 'bottom_left'],
          ],
        ],
      ],
      '#min' => 0,
      '#max' => 1025,
    ];

    $form['button']['right'] = [
      '#type' => 'number',
      '#title' => $this->t('Right'),
      '#description' => $this->t('This property controls how the button is positioned relative to right of the page.'),
      '#default_value' => $config['right'] ?? '10',
      '#states' => [
        'visible' => [
          ':input[name="settings[button][position]"]' => [
            ['value' => 'bottom_right'],
            ['value' => 'top_right'],
          ],
        ],
      ],
      '#min' => 0,
      '#max' => 1025,
    ];

    $form['button']['left'] = [
      '#type' => 'number',
      '#title' => $this->t('Left'),
      '#description' => $this->t('This property controls how the button is positioned relative to left of the page.'),
      '#default_value' => $config['left'] ?? '0',
      '#states' => [
        'visible' => [
          ':input[name="settings[button][position]"]' => [
            ['value' => 'bottom_left'],
            ['value' => 'top_left'],
          ],
        ],
      ],
      '#min' => 0,
      '#max' => 1025,
    ];

    $form['button']['width'] = [
      '#type' => 'number',
      '#title' => $this->t('Button width'),
      '#description' => $this->t('This property controls the button width.'),
      '#default_value' => $config['width'] ?? '50',
      '#min' => 0,
      '#max' => 280,
    ];

    $form['button']['unit'] = [
      '#type' => 'select',
      '#title' => $this->t('Measurement unit'),
      '#options' => [
        'px' => $this->t('px'),
        'em' => $this->t('em'),
        'rem' => $this->t('rem'),
        '%' => $this->t('%'),
      ],
      '#default_value' => $config['unit'] ?? 'px',
      '#description' => $this->t('Unit of measurement for the button position relative to the page.'),
    ];

    $form['appearance'] = [
      '#type' => 'details',
      '#title' => $this->t('Button appearance configuration'),
      '#open' => TRUE,
      '#description' => $this->t('Configure button appearance on the page'),
    ];

    $form['appearance']['stroke_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Stroke color'),
      '#default_value' => $config['stroke_color'] ?? '#fff',
      '#required' => TRUE,
      '#description' => $this->t('Button stroke color'),
    ];

    $form['appearance']['stroke_width'] = [
      '#type' => 'number',
      '#title' => $this->t('Stroke width'),
      '#default_value' => $config['stroke_width'] ?? 3,
      '#required' => TRUE,
      '#min' => 1,
      '#max' => 50,
      '#description' => $this->t('Button stroke width'),
    ];

    $form['active'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Active'),
      '#default_value' => $config['active'] ?? TRUE,
      '#description' => $this->t('Enable full screen mode'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    $position = $values['button']['position'] ?? 'top_right';
    $top = $values['button']['top'] ?? 0;
    $right = $values['button']['right'] ?? 0;
    $bottom = $values['button']['bottom'] ?? 0;
    $left = $values['button']['left'] ?? 0;
    $width = $values['button']['width'] ?? '50';
    $unit = $values['button']['unit'] ?? 'px';
    $stroke_width = $values['appearance']['stroke_width'] ?? 0;
    $stroke_color = $values['appearance']['stroke_color'] ?? 1;
    $active = $values['active'] ?? FALSE;

    $this->configuration['position'] = $position;
    $this->configuration['top'] = $top;
    $this->configuration['right'] = $right;
    $this->configuration['bottom'] = $bottom;
    $this->configuration['left'] = $left;
    $this->configuration['width'] = $width;
    $this->configuration['unit'] = $unit;
    $this->configuration['stroke_width'] = $stroke_width;
    $this->configuration['stroke_color'] = $stroke_color;
    $this->configuration['active'] = $active;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $config = $this->getConfiguration();
    $active = $config['active'] ?? FALSE;
    if ($active) {
      $full_screen_mode_data = [];

      $position = $config['position'] ?? 'top_right';
      $top = $config['top'] ?? '0';
      $right = $config['right'] ?? '0';
      $bottom = $config['bottom'] ?? '0';
      $left = $config['left'] ?? '0';
      $width = $config['width'] ?? '50';
      $unit = $config['unit'] ?? 'px';
      $stroke_width = $config['stroke_width'] ?? '3';
      $stroke_color = $config['stroke_color'] ?? '#000';

      if ($position == 'top_right') {
        $buton_array['top'] = $top . $unit;
        $buton_array['right'] = $right . $unit;

      }
      elseif ($position == 'top_left') {
        $buton_array['top'] = $top . $unit;
        $buton_array['left'] = $left . $unit;

      }
      elseif ($position == 'bottom_right') {
        $buton_array['bottom'] = $bottom . $unit;
        $buton_array['right'] = $right . $unit;

      }
      else {
        $buton_array['bottom'] = $bottom . $unit;
        $buton_array['left'] = $left . $unit;
      }

      $build = [
        '#cache' => [
          'max-age' => 0,
        ],
      ];

      $path_css = " stroke: {$stroke_color};
      stroke-width: {$stroke_width}";

      $unit = ($unit != '%') ? $unit : 'px;';

      $svg_css = "width: {$width}{$unit};";

      $btn_class = $this->arrayToCss($buton_array);
      $full_screen_mode_data = [
        'btn_class' => $btn_class,
        'path_css' => $path_css,
        'svg_css' => $svg_css,
      ];

      $build['full_screen_mode'] = [
        '#theme' => 'full_screen_mode',
        '#full_screen_mode_data' => $full_screen_mode_data,
      ];

      $build['#attached']['library'][] = 'full_screen_mode/full_screen_mode';

      return $build;
    }
    return [];
  }

  /**
   * Function to convert an array to a CSS class string.
   */
  public function arrayToCss($cssArray) {
    $cssRule = '';

    foreach ($cssArray as $property => $value) {
      $cssRule .= $property . ': ' . $value . ';';
    }

    return $cssRule;
  }

}
