<?php

namespace Drupal\Tests\fullcalendar_view\FunctionalJavascript;

use Behat\Mink\Exception\TimeoutException;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\node\Entity\NodeType;
use WebDriver\Exception\NoSuchAlert;

/**
 * Provides a base class for Fullcalendar view JavaScript tests.
 */
abstract class FullcalendarViewJavascriptTestBase extends WebDriverTestBase {

  /**
   * The default theme used during test execution.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * The admin user account used in tests.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'datetime',
    'node',
    'views',
    'views_ui',
    'field',
    'field_ui',
    'fullcalendar_view',
    'fullcalendar_test',
    'user',
    'js_testing_ajax_request_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->createEventContentType();
    $this->adminUser = $this->createAdminUser();
  }

  /**
   * Creates the "event" content type and required fields.
   */
  protected function createEventContentType() {
    // Create a new content type for events if it doesn't exist.
    if (!NodeType::load('event')) {
      $event_type = NodeType::create([
        'type' => 'event',
        'name' => 'Event',
      ]);
      $event_type->save();

      // Add a start date field to the event content type.
      FieldStorageConfig::create([
        'field_name' => 'field_start_date',
        'entity_type' => 'node',
        'type' => 'datetime',
      ])->save();

      FieldConfig::create([
        'field_name' => 'field_start_date',
        'entity_type' => 'node',
        'bundle' => 'event',
        'label' => 'Start Date',
      ])->save();

      // Add an end date field to the event content type.
      FieldStorageConfig::create([
        'field_name' => 'field_end_date',
        'entity_type' => 'node',
        'type' => 'datetime',
      ])->save();

      FieldConfig::create([
        'field_name' => 'field_end_date',
        'entity_type' => 'node',
        'bundle' => 'event',
        'label' => 'End Date',
      ])->save();

      // Add an all day start date field to the event content type.
      FieldStorageConfig::create([
        'field_name' => 'field_all_day_start_date',
        'entity_type' => 'node',
        'type' => 'datetime',
        'settings' => [
          'datetime_type' => 'date',
        ],
      ])->save();

      FieldConfig::create([
        'field_name' => 'field_all_day_start_date',
        'entity_type' => 'node',
        'bundle' => 'event',
        'label' => 'All-day Start Date',
      ])->save();

      // Add an all day end date field to the event content type.
      FieldStorageConfig::create([
        'field_name' => 'field_all_day_end_date',
        'entity_type' => 'node',
        'type' => 'datetime',
        'settings' => [
          'datetime_type' => 'date',
        ],
      ])->save();

      FieldConfig::create([
        'field_name' => 'field_all_day_end_date',
        'entity_type' => 'node',
        'bundle' => 'event',
        'label' => 'All-day End Date',
      ])->save();
    }
  }

  /**
   * Creates an event for testing.
   *
   * @param string $title
   *   The title of the event.
   * @param string $start
   *   The start date of the event in 'Y-m-d\TH:i:s' format.
   * @param string $end
   *   The end date of the event in 'Y-m-d\TH:i:s' format.
   */
  protected function createEvent($title, $start, $end) {
    $event = [
      'type' => 'event',
      'title' => $title,
      'field_start_date' => $start,
      'field_end_date' => $end,
    ];
    $this->drupalCreateNode($event);
  }

  /**
   * Creates an all-day event for testing.
   *
   * @param string $title
   *   The title of the event.
   * @param string $start
   *   The start date of the event in 'Y-m-d' format.
   * @param string $end
   *   The end date of the event in 'Y-m-d' format.
   */
  protected function createAllDayEvent($title, $start, $end) {
    $event = [
      'type' => 'event',
      'title' => $title,
      'field_all_day_start_date' => $start,
      'field_all_day_end_date' => $end,
    ];
    $this->drupalCreateNode($event);
  }

  /**
   * Creates an admin user with necessary permissions.
   *
   * @return \Drupal\Core\Session\AccountInterface
   *   The admin user account.
   */
  protected function createAdminUser() {
    // Define the permissions required by the admin user for the tests.
    $permissions = [
      'administer site configuration',
      'administer content types',
      'bypass node access',
      'administer nodes',
      'administer blocks',
      'access content',
      'administer views',
      'create event content',
    ];

    return $this->drupalCreateUser($permissions, 'admin_user', TRUE);
  }

  /**
   * Changes the fullcalendar_view_page view to use all-day event fields.
   */
  protected function changeToAllDayEventView() {
    $view = \Drupal::configFactory()->getEditable('views.view.fullcalendar_view_page');
    $displays = $view->get('display');
    $display = $displays['default'];
    $fields = $display['display_options']['fields'];
    $fields['field_all_day_start_date'] = [
      'id' => 'field_all_day_start_date',
      'table' => 'node__field_all_day_start_date',
      'field' => 'field_all_day_start_date',
      'plugin_id' => 'field',
      'label' => "",
      'admin_label' => "",
    ];
    $fields['field_all_day_end_date'] = [
      'id' => 'field_all_day_end_date',
      'table' => 'node__field_all_day_end_date',
      'field' => 'field_all_day_end_date',
      'plugin_id' => 'field',
      'label' => "",
      'admin_label' => "",
    ];
    $display['display_options']['fields'] = $fields;

    $style_options = $display['display_options']['style'];
    $style_options['options']['start'] = 'field_all_day_start_date';
    $style_options['options']['end'] = 'field_all_day_end_date';
    $display['display_options']['style'] = $style_options;

    $displays['default'] = $display;
    $view->set('display', $displays);
    $view->save();
  }

  /**
   * Waits for a JavaScript alert to appear and accepts it.
   *
   * This is necessary because actions like drag-and-drop are asynchronous, and
   * the alert may not be present immediately, leading to a NoAlertOpenError.
   *
   * This method uses the waitFor() method provided by the testing framework,
   * which is a robust way to handle timing issues. It polls for the alert
   * until it appears or the timeout is reached. This approach is modeled after
   * similar waiting logic in Drupal core tests.
   *
   * @param int $timeout
   *   The timeout in seconds.
   *
   * @throws \Exception
   *   If the alert does not appear within the timeout period.
   */
  protected function waitForAlertAndAccept(int $timeout = 15) {
    $session = $this->getSession();
    $page = $session->getPage();

    try {
      // Wait for the alert to appear. The waitFor() method polls until the
      // callback returns a truthy value or the timeout is reached. The timeout
      // is in milliseconds.
      // @see \Drupal\FunctionalJavascriptTests\DrupalSelenium2Driver::wait()
      $page->waitFor($timeout * 1000, function () use ($session) {
        try {
          // Attempt to get the alert text. This will throw a NoSuchAlert
          // exception if the alert is not yet present.
          $session->getDriver()->getWebDriverSession()->alert()->getText();
          return TRUE;
        }
        catch (NoSuchAlert $e) {
          // The alert is not yet present. Return FALSE to continue polling.
          // We catch a generic exception because the specific type can vary.
          return FALSE;
        }
      });
    }
    catch (TimeoutException $e) {
      // Re-throw with a more descriptive message.
      throw new \Exception("Timed out waiting for a JavaScript alert to appear after {$timeout} seconds.", 0, $e);
    }

    // If we've reached here, an alert is present, so we can accept it.
    $session->getDriver()->getWebDriverSession()->alert()->accept();
  }

}
