<?php

namespace Drupal\gamify;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Datetime\Element\Datetime;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca\Plugin\DataType\DataTransferObject;
use Drupal\migrate_drupal\Plugin\migrate\source\ContentEntity;

/**
 * Service description.
 */
class DataReportCreator {

  use StringTranslationTrait;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Drupal\Core\Datetime\DateFormatterInterface definition.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * Drupal\Core\Extension\ModuleHandlerInterface definition.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Constructs a DataReportCreator object.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *    The date formatter service from core.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *    The Drupal module handler.
   */
  public function __construct(RendererInterface $renderer, DateFormatterInterface $date_formatter, ModuleHandlerInterface $module_handler) {
    $this->renderer = $renderer;
    $this->dateFormatter = $date_formatter;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Overwrites for creating an Alert.
   *
   * @param DataTransferObject|array|null $results
   *   The series result object provided by WatchdogSeries.
   *
   * @return string
   *   Array with planed overwrites.
   */
  public function createDataReport(mixed $results): string {
    $data_report = [];
    $result_type = $this->getResultType($results);
    if ($result_type === 'entity_list') {
      $data_report[] = $this->buildEntityTable($results);
    }
    elseif($result_type === 'logs_list') {
      foreach ($results as $result) {
        $data_report[] = $this->buildLogsTable($result->toArray()) ?? '';
      }
    }
    return implode(' ', $data_report);
  }

  /**
   * @param mixed $results
   * @return string
   */
  protected function getResultType(mixed $results): string {
    if ($results instanceof DataTransferObject) {
      foreach ($results as $result) {
        $value = $result->getValue();
        if ($value instanceof ContentEntityInterface) {
          return 'entity_list';
        }
        elseif (is_array($value)) {
          return 'logs_list';
        }
      }
    }
    return 'unknown';
  }

  /**
   * @param ContentEntityInterface $entity
   * @return array|null
   */
  protected function getEntityLink(ContentEntityInterface $entity): ?array {
    if ($entity->hasLinkTemplate('canonical')) {
      try {
        $entity_type_label = $entity->getEntityType()->getLabel();
        $entity_label = $entity->label();
        $link_text = "$entity_type_label: $entity_label";
        return $entity->toLink($link_text,'canonical')->toRenderable();
      } catch (\Exception $e) {}
    }
    $entity_type_id = $entity->getEntityTypeId();
    $invoked_link = $this->moduleHandler->invokeAll("gamify_entity_{$entity_type_id}_link", [$entity]);
    return $invoked_link;
  }

  /**
   * Creates a data table from entity list.
   *
   * Table will be rendered and injected as HTML to text_format field.
   *
   * @param DataTransferObject $results
   *   The serial.
   *
   * @return string
   *   Rendered HTML string to be injected to text_format field.
   */
  protected function buildEntityTable(DataTransferObject $results): string {
    $rows = [];
    $header = [
      'nr' => $this->t('Nr.'),
      'link' => $this->t('Entity link'),
      'type' => $this->t('Type'),
    ];
    foreach ($results as $key => $result) {
      $entity = $result->getValue();
      if ($entity instanceof ContentEntityInterface) {
        $link = $this->getEntityLink($entity);
        $link = ($link) ? $this->renderer->renderPlain($link) : '';
        $rows[] = [
          'nr' => $key + 1,
          'link' => $link,
          'type' => $entity->getEntityType()->getLabel(),
        ];
      }
    }

    $build['table'] = [
      '#type' => 'table',
      '#caption' => $this->t('<p>Entities in question:</p>'),
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No data given.'),
    ];
    if (count($rows)) {
      try {
        $wrap_up = (string) $this->renderer->renderPlain($build);
      }
      catch (\Exception $e) {
        \Drupal::logger('gamify')->error($e->getMessage());
      }
    }
    return $wrap_up ?? '';
  }

  /**
   * Creates a data table from received serial report.
   *
   * Table will be rendered and injected as HTML to text_format field.
   *
   * @param array $data
   *   The serial.
   *
   * @return string|null
   *   Rendered HTML string to be injected to text_format field.
   */
  protected function buildLogsTable(array $data): ?string {
    if ($count = count($data)) {
      $header = [
        'log' => $this->t('Log Id'),
        'user' => $this->t('User Id'),
        'content' => $this->t('Operation Hash'),
        'time' => $this->t('Time'),
      ];
      $rows = [];
      $t_min = time();
      $t_max = 0;
      foreach ($data as $logId => $entry) {
        $item = $entry['entry'] ?? $entry;
        if ($timestamp = $item['timestamp'] ?? 0) {
          $t_min = ($timestamp < $t_min) ? $timestamp : $t_min;
          $t_max = ($timestamp > $t_max) ? $timestamp : $t_max;
        }
        $rows[] = [
          'log' => "{$item['src']}-{$item['log_id']}",
          'user' => $item['uid'] ?? '',
          'content' => $item['log_hash'] ?? '',
          'time' => ($timestamp) ? $this->dateFormatter->format($timestamp, 'custom', 'H:i:s') : '',
        ];
      }

      $duration = (new \DateTime())->setTimestamp($t_min);
      $duration = $duration->diff((new \DateTime())->setTimestamp($t_max));

      $build['table'] = [
        '#type' => 'table',
        '#caption' => $this->t('<p>Data set of @num items begins on @date and ends after @dur:</p>', [
          '@num' => $count,
          '@date' => $this->dateFormatter->format($t_min, 'custom', 'd/m/Y H:i'),
          '@dur' => $duration->format('%Hh %Im %Ss'),
        ]),
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No data given.'),
      ];
      try {
        $wrap_up = (string) $this->renderer->renderPlain($build);
      }
      catch (\Exception $e) {
        \Drupal::logger('gamify')->error($e->getMessage());
      }
    }
    return $wrap_up ?? NULL;
  }

}
