<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\gamify\AlertBuilderService;
use Drupal\gamify\DataReportCreator;
use Drupal\gamify\TypedData\Options\AlertTemplateOptions;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Create a Gamify alert from template.
 *
 * @Action(
 *   id = "gamify_alert_create",
 *   label = @Translation("Gamify: Create alert from template.")
 * )
 */
class AlertCreate extends ConfigurableActionBase {

  /**
   * Drupal\gamify\DataReportCreator definition.
   *
   * @var \Drupal\gamify\DataReportCreator
   */
  protected DataReportCreator $reportCreator;

  /**
   * Drupal\gamify\AlertBuilderService definition.
   *
   * @var \Drupal\gamify\AlertBuilderService
   */
  protected AlertBuilderService $alertBuilder;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var AlertCreate $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->setDataReportCreator($container->get('gamify.data_report_creator'));
    $instance->setAlertBuilder($container->get('gamify.alert_builder'));
    return $instance;
  }


  public function execute(): void {
    $template = $this->configuration['alert_template'];

    $user_key = $this->configuration['addressed_user'];
    $user = $user_key ? $this->tokenServices->getTokenData($this->configuration['addressed_user']) : NULL;

    if ($user instanceof UserInterface) {
      $result_key = $this->configuration['result_data'] ?? NULL;
      $result_data = $result_key ? $this->tokenServices->getTokenData($result_key) : [];
      $data_report = $this->reportCreator->createDataReport($result_data);

      $this->alertBuilder->createAlertFromTemplate($template, $user, $data_report);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array
  {
    return [
        'addressed_user' => NULL,
        'alert_template' => '',
        'result_data' => NULL,
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array
  {
    $form['addressed_user'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Addressed user'),
      '#description' =>  $this->t("Add a token for the user that will receive the alert."),
      '#default_value' => $this->configuration['addressed_user'],
      '#required' => TRUE,
    ];
    $form['alert_template'] = [
      '#type' => 'select',
      '#title' => $this->t('Alert template'),
      '#description' =>  $this->t("Select the template containing the placeholder text and pre-config options."),
      '#default_value' => $this->configuration['alert_template'],
      '#options' => AlertTemplateOptions::getOptions(),
      '#required' => TRUE,
    ];
    $form['result_data'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result data'),
      '#description' =>  $this->t("Enter token from where to get the result data."),
      '#default_value' => $this->configuration['result_data'],
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['addressed_user'] = $form_state->getValue('addressed_user');
    $this->configuration['alert_template'] = $form_state->getValue('alert_template');
    $this->configuration['result_data'] = $form_state->getValue('result_data');

    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * Set Drupal\gamify\DataReportCreator definition.
   *
   * @param \Drupal\gamify\DataReportCreator $report_creator
   *   Report creator that builds a HTML table from result arrays.
   */
  public function setDataReportCreator(DataReportCreator $report_creator): void {
    $this->reportCreator = $report_creator;
  }

  /**
   * Set Drupal\gamify\DataReportCreator definition.
   *
   * @param \Drupal\gamify\AlertBuilderService $alert_builder
   *   Service that creates Gamify-alerts.
   */
  public function setAlertBuilder(AlertBuilderService $alert_builder): void {
    $this->alertBuilder = $alert_builder;
  }

}
