<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Get timestamp from last run (and optional set current).
 *
 * @Action(
 *   id = "gamify_last_run_timestamp",
 *   label = @Translation("Gamify: Get timestamp from last run (and optional update to current).")
 * )
 */
class LastRunTimestamp extends ConfigurableActionBase {

  const KEY_VALUE_COLLECTION = 'gamify-last-run';

  /**
   * Drupal\Core\KeyValueStore\KeyValueStoreInterface definition.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreInterface|mixed
   */
  protected KeyValueStoreInterface $keyValueStore;

  /**
   * Drupal\Component\Datetime\TimeInterface definition.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\gamify\Plugin\Action\LastRunTimestamp $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->setTime($container->get('datetime.time'));
    $instance->setKeyValueStore($container->get('keyvalue'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(?EntityInterface $entity = NULL): void {
    if ($key = $this->configuration['storage_key']) {
      $last_run = (int) $this->keyValueStore->get($key);
      // 0 => unset => first run. To have true-ish result set to begin of Drupal time.
      $last_run = $last_run ?: 1000000000;

      if ($token_name = $this->configuration['token_name']) {
        $this->tokenServices->addTokenData($token_name, $last_run);
      }
      if ($this->configuration['update']) {
        $now = $this->time->getRequestTime();
        $this->keyValueStore->set($key, $now);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array
  {
    return [
        'storage_key' => '',
        'update' => TRUE,
        'token_name' => '',
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array
  {
    $form['storage_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Storage key'),
      '#description' =>  $this->t("Key for storage, where timestamp of last run is saved. Using same key in different models can confuse your results."),
      '#default_value' => $this->configuration['storage_key'],
      '#required' => TRUE,
    ];
    $form['update'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Update'),
      '#description' =>  $this->t("Should the timestamp be updated to the current runtime? (Useful for debugging to disable update.)"),
      '#default_value' => $this->configuration['update'],
    ];
    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' =>  $this->t("Enter a token name to use last run timestamp in subsequent actions."),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['storage_key'] = $form_state->getValue('storage_key');
    $this->configuration['update'] = $form_state->getValue('update');
    $this->configuration['token_name'] = $form_state->getValue('token_name');

    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * Set Drupal\Component\Datetime\TimeInterface definition.
   *
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   Drupal time service.
   */
  public function setTime(TimeInterface $time): void {
    $this->time = $time;
  }

  /**
   * Set Drupal\Core\KeyValueStore\KeyValueStoreInterface definition.
   *
   * @param \Drupal\Core\KeyValueStore\KeyValueFactoryInterface $keyValueStore
   *   Drupal key-value-storage.
   */
  public function setKeyValueStore(KeyValueFactoryInterface $keyValueStore): void {
    $this->keyValueStore = $keyValueStore->get(self::KEY_VALUE_COLLECTION);;
  }

}
