<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\gamify\Traits\GamifyDbLogTrait;
use Drupal\gamify\UserPointsLogService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Query the DbLog (watchdog).
 *
 * @Action(
 *   id = "gamify_logs_db_query",
 *   label = @Translation("Gamify: Logs DB query (filtered)."),
 *   description = @Translation("Make a logs database query filtered by search string.")
 * )
 */
class LogsDbQuery extends ConfigurableActionBase {

  use GamifyDbLogTrait;

  /**
   * Drupal\gamify\UserPointsLogService definition.
   *
   * @var \Drupal\gamify\UserPointsLogService
   */
  protected UserPointsLogService $userPointsLogService;

  /**
   * The current active database's master connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static  {
    /** @var \Drupal\gamify\Plugin\Action\LogsDbQuery $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->setUserPointsLogService($container->get('gamify.user_points_log_service')) ;
    $instance->setDatabaseService($container->get('database'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $pattern = $this->configuration['search_str'] ?? NULL;
    $user_token = $this->configuration['user'] ?? NULL;
    $user = $user_token ? $this->tokenServices->getTokenData($user_token) : NULL;
    $time = $this->configuration['time'] ?: NULL;
    if ($time) {
      if (((int) $time) > 1000000000) {
        // Asume time is a timestamp. 1000000000 ~= Begin of Drupal time.
        $timestamp = (int) $time;
      }
      elseif ($datetime = strtotime($time)) {
        // Asume time is a datetime string.
        $timestamp = $datetime;
      }
      else {
        // Assume time is a token.
        $timestamp = $this->tokenServices->getTokenData($time);
        $timestamp = ($timestamp) ? $timestamp->toArray() : [];
        $timestamp = (count($timestamp)) ? (int) $timestamp[0] : NULL;
      }
      // If timestamp is set, but is not valid, we use timestamp from "now" to get no data from query.
      $time = $timestamp ?: strtotime('now');
    }
    $result_token_key = $this->configuration['result_token_key'];

    $query_result = $this->queryDbLog($pattern, 'gamify_log', $user, $time, TRUE);
    if ($pattern && $result_token_key) {
      $this->tokenServices->addTokenData($result_token_key, $query_result);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'search_str' => '',
      'user' => '',
      'time' => '',
      'result_token_key' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {

    $form['search_str'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search string(s)'),
      '#description' => $this->t("Enter search string to find dbLog entries. Multiple search conditions seperated by | pipe, negate condition with ^ carot. E.g. [update][node:|^bundle='article' - NO RegEx here!"),
      '#default_value' => $this->configuration['search_str'],
      '#required' => TRUE,
      '#weight' => 10,
    ];

    $form['user'] = [
      '#type' => 'textfield',
      '#title' => $this->t('User'),
      '#description' => $this->t('Token for user entity targeted. Leave empty to omit user filter.'),
      '#default_value' => $this->configuration['user'],
      '#weight' => 20,
    ];

    $form['time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Relative time'),
      '#description' => $this->t('Enter time for earliest log entries. Use timestamp, token or a relative time e.g. "-1 day". Leave empty to omit time.'),
      '#default_value' => $this->configuration['time'],
      '#weight' => 30,
    ];

    $form['result_token_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result token name'),
      '#description' => $this->t('Provide a token name under which the result will be available for subsequent actions.'),
      '#default_value' => $this->configuration['result_token_key'],
      '#required' => TRUE,
      '#weight' => 40,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['search_str'] = $form_state->getValue('search_str');
    $this->configuration['user'] = $form_state->getValue('user');
    $this->configuration['time'] = $form_state->getValue('time');
    $this->configuration['result_token_key'] = $form_state->getValue('result_token_key');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * Set Drupal\gamify\UserPointsLogService definition.
   *
   * @param \Drupal\gamify\UserPointsLogService $user_points_log_service
   *   The user points log service.
   */
  public function setUserPointsLogService(UserPointsLogService $user_points_log_service): void {
    $this->userPointsLogService = $user_points_log_service;
  }

  /**
   * Set Drupal\Core\Database\Connection definition.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function setDatabaseService(Connection $database): void {
    $this->database = $database;
  }

}
