<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;

/**
 * Uses RegEx to distill params from log_hash and save to result.
 *
 * This action can be used to obtain parameters for later searches from
 * previous interim results. It saves the results as an array in the result
 * token so that conditions can access them later.
 *
 * @Action(
 *   id = "gamify_logs_params_distiller",
 *   label = @Translation("Gamify: Logs params distiller (RegEx)."),
 *   description = @Translation("Uses RegEx to distill params from log_hash and save the result to be used in subsequent actions.")
 * )
 */
class LogsParamsDistiller extends ConfigurableActionBase {

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException | \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function execute(): void {
    $prev_result_key = $this->configuration['prev_result_token'];
    $prev_results = !empty($prev_result_key) ? $this->tokenServices->getTokenData($prev_result_key)->toArray() : NULL;
    $distill_pattern = $this->configuration['distill_pattern'] ?? NULL;
    $result_token_name = $this->configuration['result_token_name'] ?? NULL;

    $interim_result = [];
    if ($prev_results && $distill_pattern && $result_token_name) {
      foreach ($prev_results as $key => $prev_result) {
        $log_hash = $prev_result['log_hash'] ?? NULL;
        if (!is_string($log_hash)) {
          continue;
        }
        if (preg_match($distill_pattern, $log_hash, $matches)) {
          unset($matches[0]);
          $interim_result[$key] = [
            'interim_stack' => $prev_result_key,
            'stack_id' => $key,
            'matches' => $matches,
            'entry' => $prev_result,
          ];
        }
      }
    }
    $this->tokenServices->addTokenData($result_token_name, $interim_result);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'prev_result_token' => '',
      'distill_pattern' => '',
      'result_token_name' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['prev_result_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Data token'),
      '#description' => $this->t('Enter token for the data that gives the basis to distill params.'),
      '#default_value' => $this->configuration['prev_result_token'],
      '#required' => TRUE,
      '#weight' => 10,
    ];
    $form['distill_pattern'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Distill pattern'),
      '#description' => $this->t('Enter a RegEx containing parentheses that form parameters for subsequent actions. E.g. /^\[create]\[(a-z_)+ /'),
      '#default_value' => $this->configuration['distill_pattern'],
      '#required' => TRUE,
      '#weight' => 20,
    ];
    $form['result_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result token name'),
      '#description' => $this->t('Provide a token name under which the result will be make available for subsequent actions.'),
      '#default_value' => $this->configuration['result_token_name'],
      '#required' => TRUE,
      '#weight' => 40,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['prev_result_token'] = $form_state->getValue('prev_result_token');
    $this->configuration['distill_pattern'] = $form_state->getValue('distill_pattern');
    $this->configuration['result_token_name'] = $form_state->getValue('result_token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

}
