<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\gamify\Traits\DbLogResultTrait;


/**
 * Classify a result in three levels, represented as min values.
 *
 * @Action(
 *   id = "gamify_logs_result_length",
 *   label = @Translation("Gamify: Logs result length"),
 *   description = @Translation("Get the length of a result object. Optional this Action will also create a result object as token value, what is required to create a data report."),
 * )
 */
class LogsResultLength extends ConfigurableActionBase {

  use DbLogResultTrait;

  public function execute(): void {
    $interim_result = $this->tokenServices->getTokenData($this->configuration['eca_data'])->toArray();

    if (is_array($interim_result) && count($interim_result)) {
      $length = 0;
      $abs_item_sum = 0;
      $data = [];
      if ($this->getNestingLevel($interim_result) >= 3) {
        $params_bag = [];
        if ($params_bag_key = $this->configuration['eca_data_params'] ?? NULL) {
          $params_bag = $this->tokenServices->getTokenData($params_bag_key)->toArray();
        };

        $result_sum = (bool) $this->configuration['result_sum'];

        // Multidimensional result.
        if (count($params_bag)) {
          // Multidimensional results leaded by group item.
          foreach ($interim_result as $key => $item_group) {
            $item_count = count($item_group);
            $top_item = isset($params_bag[$key])
              ? [$key => $params_bag[$key]] : [];
            $data[$key] = array_merge($top_item, $item_group);
            $abs_item_sum += $item_count;
            if ($result_sum) {
              $length += $item_count;
            }
            else {
              $length = ($item_count > $length) ? $item_count : $length;
            }
          }
        }
        else {

          // Multidimensional results unleaded.
          foreach ($interim_result as $item_group) {
            $item_count = count($item_group);
            $data[] = $item_group;
            $abs_item_sum += $item_count;
            if ($result_sum) {
              $length = $length + $item_count;
            }
            else {
              $length = ($item_count > $length) ? $item_count : $length;
            }
          }
        }
      }
      else {
        // One dimensional result.
        $data[] = $interim_result;
        $abs_item_sum = $length = count($interim_result);
      }

      if ($result_key = $this->configuration['eca_token_name']) {
        // Set results.
        $this->tokenServices->addTokenData($result_key, $data);
      }

      if ($length_key = $this->configuration['eca_length_name']) {
        // Set results length.
        $this->tokenServices->addTokenData($length_key, $length);
      }

    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'eca_data' => '',
      'eca_data_params' => '',
      'result_sum' => TRUE,
      'eca_length_name' => '',
      'eca_token_name' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['eca_data'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result to classify'),
      '#description' => $this->t('Token for log data. Expected array of gamify log results.'),
      '#default_value' => $this->configuration['eca_data'],
      '#required' => TRUE,
      '#weight' => 10,
    ];

    $form['eca_data_params'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Params bag'),
      '#description' => $this->t('Enter token name of params bag (if used).'),
      '#default_value' => $this->configuration['eca_data_params'],
      '#weight' => 20,
    ];

    $form['result_sum'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use result sum'),
      '#description' => $this->t('If result is a multi-dimension array, use the result sum instead of the highest result.'),
      '#default_value' => $this->configuration['result_sum'],
      '#weight' => 40,
    ];

    $form['eca_length_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result length name'),
      '#description' => $this->t('Token name under which the result length will be available as integer.'),
      '#default_value' => $this->configuration['eca_length_name'],
      '#required' => TRUE,
      '#weight' => 30,
    ];

    $form['eca_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result object'),
      '#description' => $this->t('Token name under which the result object will be made available e.g. for reports.'),
      '#default_value' => $this->configuration['eca_token_name'],
      '#weight' => 40,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['eca_data'] = $form_state->getValue('eca_data');
    $this->configuration['eca_data_params'] = $form_state->getValue('eca_data_params');
    $this->configuration['result_sum'] = $form_state->getValue('result_sum');
    $this->configuration['eca_length_name'] = $form_state->getValue('eca_length_name');
    $this->configuration['eca_token_name'] = $form_state->getValue('eca_token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

}
