<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca\Plugin\DataType\DataTransferObject;
use Drupal\gamify\Traits\DbLogResultTrait;

/**
 * Classify a result in three levels, represented as min values.
 *
 * @Action(
 *   id = "gamify_result_classification",
 *   label = @Translation("Gamify: Classify result on three levels."),
 *   description = @Translation("Classification of the result to attach differentiated actions."),
 * )
 */
class ResultClassification  extends ConfigurableActionBase {

  public function execute(): void {
    // Prepare default value.
    $level = NULL;
    if ($amount = $this->tokenServices->getTokenData($this->configuration['eca_data'])) {
      $amount = (int) $amount->getString();

      // Prepare benchmarks.
      $level_1 = (int) $this->configuration['level_1'];
      $level_2 = (int) $this->configuration['level_2'];
      $level_3 = (int) $this->configuration['level_3'];
      $level_2 = ($level_2 > $level_1) ? $level_2 : NULL;
      $level_3 = ($level_3 > $level_2) ? $level_3 : NULL;

      // Set level and Return condition.
      if ($level_3 && $amount >= $level_3) {
        $level = 'level_3';
      }
      elseif ($level_2 && $amount >= $level_2) {
        $level = 'level_2';
      }
      elseif ($amount >= $level_1) {
        $level = 'level_1';
      }
    }

    $this->tokenServices->addTokenData($this->configuration['eca_token_name'], $level);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'eca_data' => '',
      'level_1' => 0,
      'level_2' => 0,
      'level_3' => 0,
      'eca_token_name' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['eca_data'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result to classify'),
      '#description' => $this->t('Token for data. Expected is an integer values; if array is provided, the array length will be used.'),
      '#default_value' => $this->configuration['eca_data'],
      '#required' => TRUE,
      '#weight' => 10,
    ];

    $form['level_1'] = [
      '#type' => 'number',
      '#title' => $this->t('Level 1'),
      '#description' => $this->t('Min amount of the result to reach level 1.'),
      '#default_value' => $this->configuration['level_1'],
      '#required' => TRUE,
      '#weight' => 40,
    ];

    $form['level_2'] = [
      '#type' => 'number',
      '#title' => $this->t('Level 2'),
      '#description' => $this->t('Min amount to reach level 2 (disabled if level_2 <= level_1).'),
      '#default_value' => $this->configuration['level_2'],
      '#required' => TRUE,
      '#weight' => 40,
    ];

    $form['level_3'] = [
      '#type' => 'number',
      '#title' => $this->t('Level 3'),
      '#description' => $this->t('Min amount reach level 3 (disabled if level_3 <= level_2).'),
      '#default_value' => $this->configuration['level_3'],
      '#required' => TRUE,
      '#weight' => 40,
    ];

    $form['eca_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result token name'),
      '#description' => $this->t('Token name under which the classification will be made available for subsequent actions. Token value will be string ("level_1", "level_2" or "level_3") or NULL if result is less than level_1; returns FALSE if result not valid.'),
      '#default_value' => $this->configuration['eca_token_name'],
      '#required' => TRUE,
      '#weight' => 40,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['eca_data'] = $form_state->getValue('eca_data');
    $this->configuration['level_1'] = $form_state->getValue('level_1');
    $this->configuration['level_2'] = $form_state->getValue('level_2');
    $this->configuration['level_3'] = $form_state->getValue('level_3');
    $this->configuration['eca_token_name'] = $form_state->getValue('eca_token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

}
