<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\gamify\UserPointsLogService;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\gamify\TypedData\Options\AbstractUserOptions;

/**
 * Describes the gamify assign_userpoints action.
 *
 * @Action(
 *   id = "gamify_user_points_add",
 *   label = @Translation("Gamify: Add user points.")
 * )
 */
class UserPointsAdd extends ConfigurableActionBase {

  /**
   * Drupal\gamify\UserPointsLogService definition.
   *
   * @var \Drupal\gamify\UserPointsLogService
   */
  protected UserPointsLogService $userPointsLogService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\gamify\Plugin\Action\UserPointsAssignEntity $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->setUserPointsLogService($container->get('gamify.user_points_log_service'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $user_token = $this->configuration['addressed_user'];
    $user = $this->tokenServices->getTokenData($user_token) ?? NULL;
    if ($user instanceof UserInterface) {
      $log_msg = $this->configuration['log_message'];
      $quantity = $this->configuration['points_amount'];
      $quantity = (int) $this->tokenServices->replaceClear($quantity);
      $this->userPointsLogService->addPoints($quantity, $user, $log_msg);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array
  {
    return [
        'log_message' => '',
        'addressed_user' => AbstractUserOptions::CURRENT_USER,
        'points_amount' => 0,
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array
  {
    $form['points_amount'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Number of user points'),
      '#description' =>  $this->t("You may also use a negative number to deduct points. Tokens are allowed."),
      '#default_value' => $this->configuration['points_amount'],
    ];
    $form['addressed_user'] = [
      '#type' => 'textfield',
      '#title' => $this->t('User'),
      '#description' =>  $this->t("Token for the user that will receive user points."),
      '#default_value' => $this->configuration['addressed_user'],
      '#required' => TRUE,
    ];
    $form['log_message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Log message'),
      '#description' =>  $this->t("Log message commenting the assignment of user points."),
      '#default_value' => $this->configuration['log_message'],
      '#required' => TRUE,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['points_amount'] = $form_state->getValue('points_amount');
    $this->configuration['addressed_user'] = $form_state->getValue('addressed_user');
    $this->configuration['log_message'] = $form_state->getValue('log_message');

    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * Set Drupal\gamify\UserPointsLogService definition.
   *
   * @param \Drupal\gamify\UserPointsLogService $user_points_log_service
   *   The user points log service.
   */
  public function setUserPointsLogService(UserPointsLogService $user_points_log_service): void {
    $this->userPointsLogService = $user_points_log_service;
  }
}
