<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\gamify\UserPointsLogService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\gamify\TypedData\Options\AbstractUserOptions;
use Drupal\gamify\TypedData\Options\EntityOperationOptions as EntityOperations;

/**
 * Describes the gamify assign_userpoints action.
 *
 * @Action(
 *   id = "gamify_user_points_add_entity",
 *   label = @Translation("Gamify: Add userpoints (with entity context)."),
 *   type = "entity"
 * )
 */
class UserPointsAddEntity extends ConfigurableActionBase {

  /**
   * Drupal\gamify\UserPointsLogService definition.
   *
   * @var \Drupal\gamify\UserPointsLogService
   */
  protected UserPointsLogService $userPointsLogService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\gamify\Plugin\Action\UserPointsAddEntity $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->setUserPointsLogService($container->get('gamify.user_points_log_service'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE): AccessResultInterface|bool {
    $access_result = AccessResult::allowed();
    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

  /**
   * {@inheritdoc}
   */
  public function execute(?EntityInterface $entity = NULL): void {
    $operation_id = $this->configuration['operation_id'];
    $points_amount = $this->configuration['points_amount'];
    $addressed_user = $this->configuration['addressed_user'];
    $this->userPointsLogService->assign($operation_id, $entity, $addressed_user, $points_amount);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array
  {
    return [
        'operation_id' => '',
        'addressed_user' => AbstractUserOptions::CURRENT_USER,
        'points_amount' => 0,
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array
  {
    $form['operation_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Operation'),
      '#description' =>  $this->t("The triggering operation."),
      '#default_value' => $this->configuration['operation_id'],
      '#options' => EntityOperations::getOptions(),
      '#required' => TRUE,
    ];
    $form['addressed_user'] = [
      '#type' => 'select',
      '#title' => $this->t('Addressed user'),
      '#description' =>  $this->t("User who will receive user points."),
      '#default_value' => $this->configuration['addressed_user'],
      '#options' => AbstractUserOptions::getOptions()
    ];
    $form['points_amount'] = [
      '#type' => 'number',
      '#title' => $this->t('Number of user points'),
      '#default_value' => $this->configuration['points_amount'],
      '#min' => -100,
      '#max' => 100,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['operation_id'] = $form_state->getValue('operation_id');
    $this->configuration['addressed_user'] = $form_state->getValue('addressed_user');
    $this->configuration['points_amount'] = $form_state->getValue('points_amount');

    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * Set Drupal\gamify\UserPointsLogService definition.
   *
   * @param \Drupal\gamify\UserPointsLogService $user_points_log_service
   *   The user points log service.
   */
  public function setUserPointsLogService(UserPointsLogService $user_points_log_service): void {
    $this->userPointsLogService = $user_points_log_service;
  }
}
