<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\gamify\TypedData\Options\UserPointsTypeOptions;
use Drupal\userpoints\Service\UserPointsServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the ECA condition "Compare user-points amount".
 *
 * @Action(
 *   id = "gamify_user_points_get",
 *   label = @Translation("Gamify: Get user-points of user"),
 *   description = @Translation("Compare the total user points an entity has received."),
 * )
 */
class UserPointsGet extends ConfigurableActionBase {

  /**
   * The userpoints.points service.
   *
   * @var \Drupal\userpoints\Service\UserPointsServiceInterface
   */
  protected UserPointsServiceInterface $userPointsService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\gamify\Plugin\Action\UserPointsGet $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->setUserPointsService($container->get('userpoints.points'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $entity_token = $this->configuration['entity'] ?? NULL;;
    $entity = $entity_token ? $this->tokenServices->getTokenData($entity_token) : NULL;
    if ($entity instanceof EntityInterface) {
      $user_points_type = $this->configuration['user_points_type'];
      $user_points = $this->userPointsService->getPoints($entity, $user_points_type);

      // Set token
      $token_name = $this->configuration['token_name'];
      $this->tokenServices->addTokenData($token_name, $user_points);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
        'entity' => '',
        'user_points_type' => '',
        'token_name' => 0,
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {

    $form['entity'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Entity'),
      '#description' => $this->t('Specify the entity (token) whose user-points should be compared.'),
      '#default_value' => $this->configuration['entity'],
      '#required' => TRUE,
    ];

    $form['user_points_type'] = [
      '#type' => 'select',
      '#title' => $this->t('User points type'),
      '#default_value' => $this->configuration['user_points_type'],
      '#options' => UserPointsTypeOptions::getOptions(),
      '#required' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Enter a token name to make value available for subsequent actions.'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['entity'] = $form_state->getValue('entity');
    $this->configuration['user_points_type'] = $form_state->getValue('user_points_type');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
    parent::submitConfigurationForm($form, $form_state);
  }


  /**
   * Set the user points log service.
   *
   * @param \Drupal\gamify\UserPointsLogService $user_points_log_service
   *   The user points log service.
   */
  public function setUserPointsService(UserPointsServiceInterface $user_points_service): void {
    $this->userPointsService = $user_points_service;
  }

}
