<?php

namespace Drupal\gamify\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\gamify\UserPointsLogService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\gamify\TypedData\Options\AbstractUserOptions;
use Drupal\gamify\TypedData\Options\EntityOperationOptions as EntityOperations;

/**
 * Describes the gamify revert userpoints action.
 *
 * @Action(
 *   id = "gamify_user_points_revert_entity",
 *   label = @Translation("Gamify: Revert userpoints (with entity context)."),
 *   type = "entity"
 * )
 */
class UserPointsRevertEntity extends ConfigurableActionBase {

  /**
   * Drupal\gamify\UserPointsLogService definition.
   *
   * @var \Drupal\gamify\UserPointsLogService
   */
  protected UserPointsLogService $userPointsLogService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\gamify\Plugin\Action\UserPointsRevertEntity $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->setUserPointsLogService($container->get('gamify.user_points_log_service'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE): AccessResultInterface|bool {
    $access_result = AccessResult::allowed();
    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

  /**
   * {@inheritdoc}
   */
  public function execute(?EntityInterface $entity = NULL): void {
    $operation_id = $this->configuration['operation_id'];
    $revert_operations = $this->configuration['revert_operations'];
    $addressed_user = $this->configuration['addressed_user'];
    $revert_operations = array_map(function ($op): string { return trim($op); }, explode(',', $revert_operations));
    $this->userPointsLogService->revertUserPoints($operation_id, $entity, $addressed_user, $revert_operations);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array
  {
    return [
        'operation_id' => EntityOperations::REVERT,
        'revert_operations' => 'create, update',
        'addressed_user' => AbstractUserOptions::CURRENT_USER,
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array
  {
    $form['operation_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Operation'),
      '#description' =>  $this->t("The triggering operation."),
      '#default_value' => $this->configuration['operation_id'],
      '#options' => EntityOperations::getOptions(),
      '#required' => TRUE,
    ];
    $form['revert_operations'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Revert operations'),
      '#description' =>  $this->t("The operations to be reverted separated by comma. All user points user(s) received for these operations on current entity will be reverted."),
      '#default_value' => $this->configuration['revert_operations'],
      '#required' => TRUE,
    ];
    $form['addressed_user'] = [
      '#type' => 'select',
      '#title' => $this->t('Addressed user'),
      '#description' =>  $this->t("User who will receive user points."),
      '#default_value' => $this->configuration['addressed_user'],
      '#options' => AbstractUserOptions::getOptions()
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['operation_id'] = $form_state->getValue('operation_id');
    $this->configuration['revert_operations'] = $form_state->getValue('revert_operations');
    $this->configuration['addressed_user'] = $form_state->getValue('addressed_user');

    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * Set Drupal\gamify\UserPointsLogService definition.
   *
   * @param \Drupal\gamify\UserPointsLogService $user_points_log_service
   *   The user points log service.
   */
  public function setUserPointsLogService(UserPointsLogService $user_points_log_service): void {
    $this->userPointsLogService = $user_points_log_service;
  }
}
