<?php


namespace Drupal\gamify\Plugin\ECA\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\ECA\Condition\ConditionBase;
use Drupal\gamify\TypedData\Options\IntComparisonOperatorOptions as Operators;
use Drupal\gamify\TypedData\Options\EntityOperationOptions as EntityOperations;
use Drupal\gamify\UserPointsLogService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Earliest repeat time' condition.
 *
 * Checks the min/max time that must have passed before user can repeat the same action.
 * Returns TRUE:
 * - if time since last action is greater/smaller than given time.
 * - if it is the first time user executes this action.
 * Else it returns FALSE.
 *
 * @EcaCondition(
 *   id = "gamify_earliest_repeat_time",
 *   label = @Translation("Gamify: Earliest repeat time."),
 *   description = @Translation("Checks userpoints log for execution of same action on same entity and the min/max time that must have passed before user can repeat same action."),
 *   context_definitions = {
 *     "entity" = @ContextDefinition("entity",
 *       label = @Translation("Entity")
 *     )
 *   }
 * )
 */
class EarliestRepeatTime extends ConditionBase {

  /**
   * Drupal\gamify\UserPointsLogService definition.
   *
   * @var \Drupal\gamify\UserPointsLogService
   */
  protected UserPointsLogService $userPointsLogService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\gamify\Plugin\ECA\Condition\UserPointsActionRepeats $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->setUserPointsLogService($container->get('gamify.user_points_log_service'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(): bool {
    $entity_operation = $this->configuration['entity_operation'];
    $entity = $this->getValueFromContext('entity');
    $log_hash = $this->userPointsLogService->getLogHashSearchStr($entity_operation, $entity);

    if ($last_log_entry = $this->userPointsLogService->getLastActionExecLog($log_hash)) {
      $time_elapsed = $this->userPointsLogService->getActionExecTimeDiff($last_log_entry);
      $operator = $this->configuration['operator'];
      $comparator = $this->configuration['comparator'];
      $result = Operators::compare($time_elapsed, $operator, $comparator);
    }
    else {
      // No log entry found.
      $result = !$this->configuration['fallback'];
    }

    return $this->negationCheck($result);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'entity_operation' => EntityOperations::UPDATE,
      'operator' => Operators::EQ,
      'comparator' => 0,
      'fallback' => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['entity_operation'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity operation'),
      '#default_value' => $this->configuration['entity_operation'],
      '#options' => EntityOperations::getOptions(),
      '#required' => TRUE,
    ];
    $form['operator'] = [
      '#type' => 'select',
      '#title' => $this->t('Operator'),
      '#default_value' => $this->configuration['operator'],
      '#options' => Operators::getOptions(),
      '#required' => TRUE,
    ];
    $form['comparator'] = [
      '#type' => 'number',
      '#title' => $this->t('Relative min/max time (sec.)'),
      '#description' => $this->t('Min/max time that must have passed before user can repeat the same action. (E.g if time is on day, then enter 60 * 60 * 24 = 86400)'),
      '#default_value' => $this->configuration['comparator'] ?? 0,
    ];
    $form['fallback'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('FALSE if first Time'),
      '#description' => $this->t('If checked, condition returns FALSE when user executed action never before.'),
      '#default_value' => $this->configuration['fallback'],
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['entity_operation'] = $form_state->getValue('entity_operation');
    $this->configuration['operator'] = $form_state->getValue('operator');
    $this->configuration['comparator'] = $form_state->getValue('comparator');
    $this->configuration['fallback'] = !empty($form_state->getValue('use_yaml'));
    parent::submitConfigurationForm($form, $form_state);
  }


  /**
   * Set the user points log service.
   *
   * @param \Drupal\gamify\UserPointsLogService $user_points_log_service
   *   The user points log service.
   */
  public function setUserPointsLogService(UserPointsLogService $user_points_log_service): void {
    $this->userPointsLogService = $user_points_log_service;
  }

}
