<?php


namespace Drupal\gamify\Plugin\ECA\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\ECA\Condition\ConditionBase;
use Drupal\gamify\TypedData\Options\EntityOperationOptions as EntityOperations;
use Drupal\gamify\TypedData\Options\AbstractUserOptions;
use Drupal\gamify\UserPointsLogService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the ECA condition "Max action repeats".
 *
 * @EcaCondition(
 *   id = "gamify_user_points_action_repeats",
 *   label = @Translation("Gamify: Max action repeats"),
 *   description = @Translation("Limit the number how often an user is allowed to repeated same action and receive user points.)"),
 *   context_definitions = {
 *     "entity" = @ContextDefinition("entity",
 *       label = @Translation("Entity")
 *     )
 *   }
 * )
 */
class MaxActionRepeats extends ConditionBase {

  /**
   * Drupal\gamify\UserPointsLogService definition.
   *
   * @var \Drupal\gamify\UserPointsLogService
   */
  protected UserPointsLogService $userPointsLogService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\gamify\Plugin\ECA\Condition\UserPointsActionRepeats $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->setUserPointsLogService($container->get('gamify.user_points_log_service'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(): bool {
    $entity = $this->getValueFromContext('entity');
    $entity_operation = $this->configuration['entity_operation'];
    $abstract_user = $this->configuration['abstract_user'];
    $repeats = $this->userPointsLogService->userActionRepeat($entity, $entity_operation, $abstract_user);
    if (is_int($repeats)) {
      $comparator = $this->configuration['comparator'];
      $result = $repeats <= $comparator;
      return $this->negationCheck($result);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'entity_operation' => EntityOperations::UPDATE,
      'abstract_user' => AbstractUserOptions::CURRENT_USER,
      'comparator' => 0,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {

    $form['entity_operation'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity operation'),
      '#default_value' => $this->configuration['entity_operation'],
      '#options' => EntityOperations::getOptions(),
      '#required' => TRUE,
    ];

    $form['abstract_user'] = [
      '#type' => 'select',
      '#title' => $this->t('User to verify'),
      '#default_value' => $this->configuration['abstract_user'],
      '#options' => AbstractUserOptions::getOptions(),
      '#required' => TRUE,
    ];

    $form['comparator'] = [
      '#type' => 'number',
      '#title' => $this->t('Max. entity operation repeats'),
      '#default_value' => $this->configuration['comparator'] ?? 0,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['entity_operation'] = $form_state->getValue('entity_operation');
    $this->configuration['abstract_user'] = $form_state->getValue('abstract_user');
    $this->configuration['comparator'] = $form_state->getValue('comparator');
    parent::submitConfigurationForm($form, $form_state);
  }


  /**
   * Set the user points log service.
   *
   * @param \Drupal\gamify\UserPointsLogService $user_points_log_service
   *   The user points log service.
   */
  public function setUserPointsLogService(UserPointsLogService $user_points_log_service): void {
    $this->userPointsLogService = $user_points_log_service;
  }

}
