<?php

namespace Drupal\gamify\Plugin\ECA\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca\Plugin\ECA\Condition\ConditionBase;
use Drupal\votingapi\VoteInterface;

/**
 * Plugin implementation of the ECA condition "Gamify: voted entity has user comments".
 *
 * @EcaCondition(
 *   id = "gamify_voted_entity_has_comments",
 *   label = @Translation("Gamify: voted entity has user comments"),
 *   description = @Translation("Validates TRUE if the voting user commented the voted entity OR if comments field NOT exists.)"),
 *   context_definitions = {
 *     "entity" = @ContextDefinition("entity", label = @Translation("Entity"))
 *   }
 * )
 */
class VotedEntityHasComments extends ConditionBase {

  use StringTranslationTrait;

  /**
   * Check if voted entity type has a comment field.
   *
   * @param \Drupal\votingapi\VoteInterface $vote
   *   Supplied vote entity to evaluate by plugin.
   *
   * @see \Drupal\vote\TypedData\Options\VoteResultQuality
   *
   * @return bool
   *   The evaluation of the condition.
   */
  public function evaluate(): bool {
    $vote = $this->getValueFromContext('entity');
    $user_token = $this->configuration['user'];
    $user = $this->tokenServices->getTokenData($user_token);

    $response = FALSE;
    if ($user && ($vote instanceof VoteInterface)) {
      $voted_type = $vote->getVotedEntityType();
      try {
        /** @var \Drupal\comment\CommentManagerInterface $comment_manager */
        $comment_manager = \Drupal::service('comment.manager');
        $field_exists = FALSE;
        if ($comment_fields = $comment_manager->getFields($voted_type)) {
            // Check if field_comments (or equal) exists.
            $voted_id = $vote->getVotedEntityId();
            $voted_entity = \Drupal::entityTypeManager()->getStorage($voted_type)->load($voted_id);
            $voted_bundle = $voted_entity->bundle();
            foreach ($comment_fields as $definition) {
              if (in_array($voted_bundle, $definition['bundles'])) {
                $field_exists = TRUE;
                break;
              }
            }
            // Check if user commented the voted entity.
            if (!$field_exists) {
              $response = TRUE;
            }
            else {
              $db = \Drupal::database();
              $query = $db->select('comment_field_data', 't')
                ->fields('t')
                ->condition('status', 1)
                ->condition('uid', $user->id())
                ->condition('entity_type', $voted_type)
                ->condition('entity_id', $voted_id)
                ->orderBy('cid', 'DESC');

              $query_count = $query
                ->execute()
                ->fetchAll();
              if (count($query_count)) {
                $response = TRUE;
              }
            }
        }
      }
      catch (\Exception $e) {
        \Drupal::logger('vote')->error($e->getMessage());
      }
    }
    return $this->negationCheck($response);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
        'user' => NULL,
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {

    $form['user'] = [
      '#type' => 'textfield',
      '#title' => $this->t('User'),
      '#default_value' => $this->configuration['user'],
      '#required' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['user'] = $form_state->getValue('user');
    parent::submitConfigurationForm($form, $form_state);
  }

}
