<?php

namespace Drupal\gamify\TypedData\Options;

use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Options provider to return a choice of data comparison operators.
 */
class IntComparisonOperatorOptions extends OptionsProviderBase {

  const LT = '<';
  const LTE = '<=';
  const EQ = '==';
  const GTE = '>=';
  const GT = '>';

  /**
   * {@inheritdoc}
   */
  public function getPossibleOptions(AccountInterface $account = NULL): array {
    return self::getOptions();
  }


  public static function getOptions(): array {
    return [
      self::LT => new TranslatableMarkup('< (less than)'),
      self::LTE => new TranslatableMarkup('<= (less than/equal)'),
      self::EQ => new TranslatableMarkup('== (equal)'),
      self::GTE => new TranslatableMarkup('>= (greater than/equal)'),
      self::GT => new TranslatableMarkup('> (greater than)'),
    ];
  }

  /**
   * Compare to values based on given options.
   *
   * @param int $dyn
   *   The dynamic property to compare.
   * @param int $static
   *   The static value used as benchmark.
   * @param string $operator
   *   The comparison operator.
   *
   * @return bool|null
   *   Returns the result of comparison.
   */
  public static function compare(int $dyn, string $operator, int $static): ?bool {
    if ($operator == self::LT ) {
      return $dyn < $static;
    }
    elseif ($operator == self::LTE) {
      return $dyn <= $static;
    }
    elseif ($operator == self::EQ ) {
      return $dyn == $static;
    }
    elseif ($operator == self::GTE) {
      return $dyn >= $static;
    }
    elseif ($operator == self::GT ) {
      return $dyn > $static;
    }
    return NULL;
  }


}
